/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */

#ifndef __QLIB_EXPOWER_AUXILIARY
#define __QLIB_EXPOWER_AUXILIARY

#include <vector>                           // for STL vectors
#include <utility>                          // STL utilities (like pair)
#include "qinterface_types.h"               // for the typedef's saga
#include "qexception.h"                     // for "quantum" exceptions
#include "qoperator.h"                      // class for quantum operators
#include "qntlib_bigint.h"                  // multiprecision integers

/* *********************************************************************
   | This is an auxiliary class whose goal is to "calculate" the gates |
   | which are necessary to build up the circuit for the "expower"     |
   | function in the quantum library. In practice, the ctor of this    |
   | class fills in a data structure with one record for each gate     |
   | (each gate is a multicontrolled phase gate); that record contains |
   | an array of integers, specifying which qubits the gate acts on,   |
   | and a floating point value, which is the phase factor of the gate.|
   | The circuit is built in such a way that the first bit in each     |
   | register passed to the circuit will be considered, as usual, the  |
   | most significant bit (MSB).                                       | 
   | ----------------------------------------------------------------- |
   | Say q="qubits" the number of qubits, p="power" and f="factor";    |
   | then the basic idea for building a circuit which transforms the   |
   | computational basis state |n> into exp(2 PI i f n^p)|n>, with n   |
   | in the range [0, 2^q[, is to expand n into its binary representa- |
   | tion sum_j a_j 2^j, with j in [0, q-1]. This gives the following  |
   | expression for exp(i F n^p), with F = (2 PI f):                   |
   |   prod_{j_1 ... j_p} exp(i F a_j_1 ... a_j_p 2^(j_1 + ... + j_p)) |
   | Each product of binary digits a_j_? is either zero or one. Indeed |
   | it is 1 only if all digits are 1s. Then, each factor in the over- |
   | all product corresponds to a multicontrolled phase gate, acting   |
   | on the p qubits indexed by j_1 ... j_p, with phase factor equal   |
   | to exp(i F 2^(j_1 + ... + j_p)). This algorithm is described in:  |
   |                                                                   |
   | A.D.Chepelianskii and D.L.Shepelyansky, "Schroedinger cat anima-  |
   | ted on a quantum computer", quant-ph/0202113.                     |
   | ----------------------------------------------------------------- |
   | With the previous construction, the number of gates is obviously  |
   | q^p, which, for a fixed p, is polynomial in q. However, this class|
   | uses an optimised construction, which involves the minimal possi- |
   | ble number of multicontrolled gates, when each gate is considered |
   | as a primitive (the decomposition into one- and two-qubit primi-  |
   | tives is performed at a later stage): two different gates do NOT  |
   | concern the same (unordered) set of qubits.                       |
   | It is easy to calculate the number of gates in this construction; |
   | if we say (q k) the binomial coefficient "q choose k", we have:   |
   |       Ngates = sum_k (q k) for k in the range [1, min(q,p)]       |
   | For q<=p this sum gives (2^q - 1), while for q >> p it can be     |
   | approximated by q^p/p!. Altough the behaviour for fixed p is the  |
   | same as before, it allows for a significant reduction of the num- |
   | ber of gates: already for p=4, as is the case in the previously   |
   | cited paper, ~95.8% of the multicontrolled phase gates are saved  |
   | in the limit of large q. For q=5 and p=4 the saving is 95.2%.     |
   | ----------------------------------------------------------------- |
   | This construction involves the multiplication of "factor" by very |
   | large integers (growing like the factorial of "power" and expo-   |
   | nential in "qubits"); since the number of significant digits in   |
   | "factor" is limited, this means that the fractional part of this  |
   | multiplication (which is what interests us, since angles are cy-  |
   | clical) can be affected by numerical noise unless we take parti-  |
   | cular care in the multiplication.                                 |
   | ----------------------------------------------------------------- |
   | (28 Mar 2003) S.Bettelli, I'm now using the big_int class for the |
   |   bignumber data type. This avoids all the worries about uncheck- |
   |   ed overflows and unrepresentable integers. The multiplication   |
   |   of a large integer with a floating point is dealt with in the   |
   |   bignumber class itself.                                         |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 May 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      28 Mar 2003         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      31 Mar 2003         |
   ********************************************************************* */
class gate_calculator {
public:
  // a typedef for a qubit address (type for "qubits")
  typedef ::qubit_address_type address_type;
  // a typedef for a power value (type for "power")
  typedef unsigned int power_type;
  // a typedef for the floating point factor
  typedef double factor_type;
public:
  // the most important ctor for this class
  gate_calculator(address_type the_qubits,
		  power_type   the_power,
		  factor_type  the_factor);
  // this method reads the gate description and creates the circuit
  Qop get_Qop(void);
private:
  // a generic unsigned integer (should contain power_type & address_type)
  typedef address_type number;
  // an unsigned integer type for storing VERY big integers
  typedef big_int bignumber;
  // a vector of integers for keeping a partition of "power"
  typedef std::vector<power_type> partition_type;
  // a vector of integers for keeping a set of qubit addresses
  typedef std::vector<address_type> selection_type;
  // each record can contain a partition and its multiplicity
  typedef std::vector<std::pair<partition_type, bignumber> > partition_list;
  // each record can contain the description of a multicontrolled phase gate
  typedef std::vector<std::pair<selection_type, factor_type> > gate_list;
  // the range of qubit indexes will be [0, qubits[
  const address_type qubits;
  // this is the number of indexes which are combined toghether in each gate
  const power_type power;
  // this is the additional factor passed to the constructor
  const factor_type factor;
  // a list of partition descriptions filled by calculate_partitions
  partition_list partitions;
  // the list of all multicontrolled phase gates to be built
  gate_list gates;
  // this vectors stores all the factorials from 0 up to power
  std::vector<bignumber> factorials;
private:
  // calculates all the part. of p into k groups (save them in "partitions")
  void calculate_partitions(number k, power_type p);
  // calculates the multicontrolled phase gate list
  void calculate_gates(number k, address_type q);
  // calculates the phase factor for a gate given an index selection
  factor_type calculate_factor(selection_type a_selection);
};

#endif  // __QLIB_EXPOWER_AUXILIARY
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
