/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __ABSTRACT_QUANTUM_SLICES_
#define __ABSTRACT_QUANTUM_SLICES_

#include <iostream>                   // C++ I/O streams
#include "qexception.h"               // for "quantum" exceptions
#include "qsegment.h"                 // for basic qubit segments
#include "qregister.h"                // for quantum registers
#include "qlist.h"                    // for the qubit list class
#include "qop_slice_basegate.h"       // for base gates' descriptions

/* *********************************************************************
   | This abstract interface class specifies the common properties of  |
   | a time slice, that is a bunch of homogeneous "primitive" software |
   | operations which should be acted in the same temporal slot.       |
   | The identity of each primitive operation in each slice can be     |
   | represented by a single integer code (operation code or opcode).  |
   | Primitives however can be parametrised by further variables.      |
   | ----------------------------------------------------------------- |
   | This abstract interface defines some common data types and the    |
   | core methods for time slices; the behaviour of these methods is   |
   | in common with composite operators, though the return type is     |
   | not; the following description applies both to time slices and to |
   | composite quantum operators (the Qop class).                      |
   |                                                                   |
   | split(head, jump)                                                 |
   |    This modifies the current quantum circuit so that the action   |
   |    on the first "head" qubits is unaltered, and the action on the |
   |    following qubits is shifted "jump" positions ahead. This means |
   |    that action on the (k+jump)-th qubit will be the same as the   |
   |    previous action on the k-th qubit. Qubits from head-th up to   |
   |    (head+jump-1)-th are left standing (i.e. they undergo a unita- |
   |    ry evolution which is represented by the identity matrix).     |
   |                                                                   |
   |    ----------------  \                                            |
   |    ----------------   | first "head" qubits are unaltered         |
   |    ----------------  /                                            |
   |    -------.    ----    \                                          |
   |    -------.\   ----     | identity run here ("jump" qubits)       |
   |    -------.\\  ----    /                                          |
   |            \\`-----      \                                        |
   |             \`-----       | circuit starts here again             |
   |              `-----      /                                        |
   |                                                                   |
   | invert(head, size)                                                |
   |    This modifies the current quantum circuit so that the action   |
   |    on the first "head" qubits and on those from "head+size"-th on |
   |    is unaltered, while the action on the qubits in the middle is  |
   |    reversed in order.                                             |
   |                                                                   |
   |    ----------------  \                                            |
   |    ----------------   | first "head" qubits are unaltered         |
   |    ----------------  /                                            |
   |    -------.   .----    \                                          |
   |    ------..\-/-----     | the following "size" qubit lines have   |
   |    ------'`-\------     | their order reversed                    |
   |    --------' `-----    /                                          |
   |    ----------------      \                                        |
   |    ----------------       | qubits from now on are unaltered      |
   |    ----------------      /                                        |
   |                                                                   |
   | remap(permutation)                                                |
   |    This modifies the current quantum circuit so that the old line |
   |    i is remapped onto the new line permutation[i]. It is a sort   |
   |    of generalisation of the split and invert operations.          |
   | adjoin()                                                          |
   |    This modifies the current quantum circuit into the adjoint     |
   |    circuit; this means that the order of execution of all time    |
   |    slices is reversed and their operations are adjoint.           |
   | offset(jump)                                                      |
   |    This moves all the quantum circuit ahead by "jump" qubits. It  |
   |    is exactly the same as split(0, jump). The method is not even  |
   |    virtual and needs not be supplied by derived classes.          |
   | operator()(a_register)                                            |
   |    This method actually runs the current operator on the supplied |
   |    quantum register. If the operator is composite this boils down |
   |    to call the corresponding method for every time slice. If it   |
   |    is a time slice a custom run method is supplied by each speci- |
   |    fic time slice. Note that not all operations can be run on     |
   |    every register!                                                |
   |                                                                   |
   | ----------------------------------------------------------------- |
   | Each slice must present a specialisation of the clone() function  |
   | so that time slice copying will be possible without knowing the   |
   | real nature of time slices. A call to clone() must return a       |
   | pointer to a dynamically allocated copy of the most derived class |
   | downcasted to (Qop_slice *).                                      |
   | ----------------------------------------------------------------- |
   | (06 Sep 2001) S.Bettelli; the field "ancillae" has been added. It |
   | declares the first ancillae qubits for the slice to be ancillae.  |
   | This makes a difference in the split and invert (& offset) calls, |
   | which ignore these qubits and start counting after them. This     |
   | field is necessary to support (multi-)controlled operations, but  |
   | can come useful for library routines too.                         |
   | ----------------------------------------------------------------- |
   | (04 Oct 2001) S.Bettelli; new pure virtual method -> control()    |
   | It forces primitive slices to supply a routine which calculates   |
   | the simply controlled version of themselves and stores it into    |
   | the passed container. Not very elegant but it allows for an easy  |
   | construction of controlled operators.                             |
   | ----------------------------------------------------------------- |
   | (05 Dec 2001) S.Bettelli; new virtual method has_adjoint_basegate |
   | which returns true if the quantum gates associated to the two     |
   | slices are adjoint. The new method is useful for the simplifica-  |
   | tion routines. Also supplied a default virtual dummy adjoin()     |
   | which does nothing (it is already enough for selfadjoint gates).  |
   | ----------------------------------------------------------------- |
   | (04 Jan 2002) S.Bettelli; Mixed Qop_slice and Qop_fixed_arity.    |
   | Now the index lists are private to this class, so that some of    |
   | the methods do not need virtualness. For instance, invert() and   |
   | split() can now be written withing Qop_slice.                     |
   | ----------------------------------------------------------------- |
   | (15 May 2002) S.Bettelli, introduced the remap() method; this is  |
   | a big simplification during circuit construction, altough every-  |
   | thing in principle can be done with split() and invert(), which   |
   | are to be preferred, when possible, since they are more efficient.|
   | ----------------------------------------------------------------- |
   | (30 Oct 2002) S.Bettelli, moved all ancilla stuff to the slice    |
   | list class, it's easier. Slices do not know about ancillae now.   | 
   | ----------------------------------------------------------------- |
   | (10 Nov 2002) S.Bettelli, introduced a pointer to a constant      |
   | Qop_slice_basegate object, which describes a base gate (exception |
   | made for a possible additional parameter). Most get functions can |
   | now be implemented using this pointer instead of virtual methods. |
   | The opcode member has been removed because of this reason. Moved  |
   | output_details to get_details and simplified a bit with operator<<|
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 23 May 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      13 May 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Oct 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      10 Nov 2002         |
   ********************************************************************* */
class Qop_slice {
  // an output function for debugging of a time slice
  friend std::ostream &operator<<(std::ostream &os, const Qop_slice &a_slice);
public:
  // this is an alias for the qubit address type
  typedef Qubit_segment::address          address;
  // this is an alias for the qubit segment size type
  typedef Qubit_segment::size_type        size_type;
  // this is an integer type for a gate index in the gate database
  typedef Qop_slice_basegate::size_type   gateidx_type;
  // this is an integer type for an operation code
  typedef Qop_slice_basegate::opcode_type opcode_type;
  // this is an alias for an index list (formal addresses)
  typedef Qubit_list                      list_type;
  // this is an alias for a collection of qubit lists
  typedef Qubit_list_collection           collection_type;
  // this is an alias for the size of a collection of qubit lists
  typedef collection_type::size_type      index_type;
  // a class for invalid (depending on the context) index lists
  _Qexception(invalid_list);
  // DEBUG!!! don't use this
  const list_type getthelist(void) const { return get_list(0); }
public:
  // this method shifts a part of the slice down
  void split(size_type head, size_type jump);
  // this method inverts the order of a part of the slice
  void invert(size_type head, size_type size);
  // this is a shortcut for a split() with zero head
  void offset(size_type jump) { split(0, jump); }
  // this method performs a permutation of the indexes in the slice
  void remap(const Qubit_list &permutation_list);
  // this method modifies the current operator into its adjoint
  virtual void adjoin(void) { /* default is self-adjoint */ }
  // force a clone() method in each derived class
  virtual Qop_slice *clone(void) const = 0;
  // force a method for building controlled circuits (horrible!)
  virtual void control(class Qop_slicelist *, size_type cfirst) const = 0;
  // this is the most important run method for a slice (virtual!)
  virtual void operator()(const Qreg &a_register) const;
public:
  // true if the two slices are one the adjoint of the other
  bool is_adjoint_of(const Qop_slice &a_slice) const;
  // true if the given address is used by the slice
  bool is_referencing(address an_address) const;
public:
  // this returns the base gate position in the gate database
  gateidx_type get_index(void)  const { return basegate->get_index(); }
  // this returns the operation code (through basegate pointer)
  opcode_type  get_opcode(void) const { return basegate->get_opcode(); }
  // this returns the name of the associated base gate
  std::string  get_name(void)   const { return basegate->get_name(); }
  // this method returns the slice parallelisation index
  size_type    get_parallelisation(void) const;
  // this returns the minimum size of a register for this slice
  size_type    get_occupation(void) const;
  // suggest a method for optional parameter passing
  virtual opcode_type get_parameter(void) const { return 0; }
  // suggest a method for additional base gate specific information
  virtual void        get_details(std::ostream &os) const { os.flush(); }
  // the destructor does nothing but will be called explicitely
  virtual ~Qop_slice() { }
protected:
  // the default constructor is explicitely prohibited
  Qop_slice();
  // standard constructor accepting preformed lists and the base gate
  Qop_slice(const Qop_slice_basegate *the_bg, const collection_type &lists);
  // this constructor accepts one preformed list and the base gate
  Qop_slice(const Qop_slice_basegate *the_bg, const list_type &user_list);
  // this constructor accepts two preformed lists and the base gate
  Qop_slice(const Qop_slice_basegate *the_bg,
	    const list_type &user_list_1, const list_type &user_list_2);
protected:
  // a routine for an alternative operator() for swap gates
  void classical_permutation(const Qreg &a_register) const;
  // true if the gates associated to the two slices are adjoint
  virtual bool has_adjoint_basegate(const Qop_slice &a_slice) const;
  // this method returns a constant reference to the N-th index list
  const list_type &get_list(index_type list_index) const;
private:
  // this method returns a reference to the N-th index list
  list_type &get_list(index_type list_index);
  // this method returns the number of index lists
  index_type get_list_number(void) const;
  // this method translates formal addresses into real addresses
  collection_type translate(const Qreg &a_reg) const;
private:
  // a pointer to the properties of the base gate
  const Qop_slice_basegate *basegate;
  // this is a container for address lists.
  collection_type index_lists;
};

/* input all the inline functions */
#include "qop_slice_inlines.h"

#endif  // __ABSTRACT_QUANTUM_SLICES_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
