/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __COMPOSITE_QUANTUM_OPERATORS_
#define __COMPOSITE_QUANTUM_OPERATORS_

#include <iostream>                   // C++ I/O streams
#include "qop_slicelist.h"            // for lists of time slices

/* *********************************************************************
   | This class is used for generic composite quantum operators.       |
   | Objects from this class can be concatenated and manipulated in    |
   | various ways, and they can be acted upon registers.               |
   | Their construction, manipulation and simplification is completely |
   | classical however. One can contact the quantum device when every- |
   | thing is already set for quantum operators. There is a longer     |
   | description of the split(), invert(), remap(), offset(), adjoin() |
   | and operator() methods in the Qop_slice class to which the reader |
   | is urged for further details.                                     |
   | ----------------------------------------------------------------- |
   |                                                                   |
   | Requirements on quantum operators                                 |
   | =================================                                 |
   |                                                                   |
   | 1)  Quantum operators which are the identity or a copy of a pre-  |
   |     existing operator can be created.                             |
   | 2)  Operators which are the controlled version of preexisting     |
   |     operators can be constructed.                                 |
   | 3)  Operators which are oracles or phase oracles corresponding to |
   |     classical mathematical mappings can be constructed (TODO !)   |
   | 4)  Operators can be concatenated (composed).                     |
   | 5)  Operators can be modified through permutations: basic permu-  |
   |     tations are split, invert, remap and offset (both the mutable |
   |     and the non mutable version).                                 |
   | 6)  Operators can be conjugated (mutable and non mutable)         |
   | 7)  Operators can be executed on registers.                       |
   |                                                                   |
   | Not really required, but useful                                   |
   | ===============================                                   |
   |                                                                   |
   | 8)  Concatenation of operators has more efficient forms, like     |
   |     augmentation and destructive augmentation.                    |
   | 9)  Operation execution can be cached and resent to the quantum   |
   |     device without translating addresses again (TODO).            |
   | 10) Depth, occupation and ancilla requirements can be inquired.   |
   | 11) Statistics about the operator structure can be displayed.     |
   |                                                                   |
   | ----------------------------------------------------------------- |
   | Quantum operators are basically a list of pointers to time slices.|
   | Each pointer points to a private time slice, which is created     |
   | dynamically and which must be freed at destruction time.          |
   | (13 May 2002) S.Bettelli, introduced the remap() transformation.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 24 May 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      13 May 2002         |
   ********************************************************************* */
class Qop {
  /* an output function for debugging */
  friend std::ostream &operator<<(std::ostream &os, const Qop &an_operator);
public:
  /* DEBUG!!! don't use this */
  Qop transform(Qop_slice::address a=0, int b=0, int c=0) const;
  // this is an alias for the qubit address type
  typedef Qubit_segment::address address;
  // this is an alias for the qubit segment size type
  typedef Qubit_segment::size_type size_type;
  // a class for problems during a non mutable permutation
  _Qexception(invalid_NM_permutation);
public:
  // the default constructor creates the identity operator
  Qop() : operations() { }
  // the copy constructor copies the internal list
  Qop(const Qop &an_operator);
  // this constructor creates a controlled operator
  Qop(const Qop &an_operator, size_type control_size);
  // the assignment operator (corresponds to list assignment)
  Qop &operator=(const Qop &an_operator);
  // the trivial virtual destructor
  virtual ~Qop() { }
  // this ctor translates a classical mapping into an oracle (TODO)
  // Qop(int (*f)(int), int in, int out);
  // this ctor translates a boolean function into a phase oracle (TODO)
  // Qop(bool (*f)(int), int in);
public:
  // concatenation operator returning a new Qop
  friend Qop operator&(const Qop &operator_1, const Qop &operator_2);
  // this is the augmentation operator
  Qop &operator&=(const Qop &an_operator);
  // this is a sort of operator&= which destroys the argument (efficiency)
  Qop &operator<<(Qop &an_operator);
  // this method executes the operator on a given register
  void operator()(const Qreg &a_register) const;
public:
  // this method shifts a part of the circuit down (mutable)
  Qop &split(size_type head, size_type jump);
  // this method inverts the order of a part of the circuit (mutable)
  Qop &invert(size_type head, size_type size);
  // this method remaps the order of qubit lines in the circuit (mutable)
  Qop &remap(const Qubit_list &permutation);
  // this method modifies the current operator into its adjoint (mutable)
  Qop &adjoin(void);
  // this is a shortcut for a split() with zero head
  Qop &offset(size_type jump) { return split(0, jump); }
public:
  // enumeration for selecting invert or split
  enum op_type { SPLIT, INVERT };
  // non mutable version of split and invert with selector
  Qop operator()(size_type arg1, size_type arg2, op_type op = SPLIT) const;
  // this returns the adjoint operator (non mutable adjoin())
  Qop operator!(void) const;
  // this returns the shifted operator (non mutable offset())
  Qop operator>>(size_type jump) const;
public:
  // this method returns the circuit depth (the number of slices)
  size_type depth(void) const { return get_operations().get_size(); }
  // this method returns the circuit footpath (min size of register)
  size_type occupation(void) const { return get_operations().get_occupation();}
  // this method returns the number of ancillae at run-time
  size_type ancillae(void) const { return get_operations().get_ancillae(); }
  // this method returns the maximum parallelisation degree for this operator
  size_type parallelisation(void) const {
    return get_operations().get_parallelisation(); }
  // print statistics about this operator to an output stream
  void print_statistics(std::ostream &os) const;
protected:
  // a protected constructor which accepts a preformed operation list
  Qop(Qop_slicelist &preformed_list);
  // a protected constructor which accepts a pointer to a Qop_slice
  Qop(Qop_slice *a_slice);
private:
  // pointer to private ordered list for constituent time slices
  Qop_slicelist operations;
  // this method returns a constant reference to the operation list
  const Qop_slicelist &get_operations(void) const { return operations; }
  // this method returns a non-constant reference to the operation list
  Qop_slicelist &get_operations(void) { return operations; }
};

/* input all the inline functions */
#include "qoperator_inlines.h"

#endif  // __COMPOSITE_QUANTUM_OPERATORS_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
