/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __COMPOSITE_QUANTUM_OPERATORS_INLINES_
#define __COMPOSITE_QUANTUM_OPERATORS_INLINES_

#include "qoperator.h"                // the class declaration

/* *********************************************************************
   | The copy constructor copies the internal list, preserving the     |
   | identity of the most derived class for each element in the list.  |
   | This is done through a call to the clone method of slices (this   |
   | is embedded in the append() call). I can't avoid supplying this   |
   | method since the compiler will otherwise do that (uncorrectly)    |
   | for me.                                                           |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 13 Jul 2001         |
   ********************************************************************* */
inline Qop::Qop(const Qop &an_operator) 
{ get_operations().append(an_operator.get_operations()); }

/* *********************************************************************
   | This is a protected constructor accepting a preformed list, which |
   | is spliced into the internal list. This method is present only    |
   | for sake of efficiency, and is used in the concatenation methods  |
   | and in some constructors for the derived classes.                 |
   | Note that the passed list will be empty after ctor has finished.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 13 Jul 2001         |
   ********************************************************************* */
inline Qop::Qop(Qop_slicelist &preformed_list)
{ get_operations().splice(preformed_list); }

/* *********************************************************************
   | This is a protected constructor accepting a pointer to a time     |
   | slice, which is saved in the internal list. This method is needed |
   | for efficiently implementing primitive operations, which only     |
   | contain one slice. It needs not being made public. If the passed  |
   | pointer is NULL, do nothing and return.                           |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 07 Sep 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Jun 2002         |
   ********************************************************************* */
inline Qop::Qop(Qop_slice *a_slice)
{ if (a_slice) get_operations().push_front(a_slice); }

/* *********************************************************************
   | This method is the assignement operator for Qops. It corresponds  |
   | to assignment between slice lists.                                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 07 Jan 2002         |
   ********************************************************************* */
inline Qop &Qop::operator=(const Qop &an_operator)
{ get_operations() = an_operator.get_operations(); return *this; }

/* *********************************************************************
   | This is the augmentation operation for quantum operators. It is   |
   | just like the concatenation operation, but instead of creating a  |
   | new Qop we are going to modify the current one (for efficiency).  |
   | The argument will be appended to the current operation list (i.e. |
   | it will be executed after the execution of the current operation  |
   | list). The list of operations subsequently undergoes a simplifi-  |
   | cation routine.                                                   |
   | (27 Sep 2001) S.Bettelli. New convention on direction of concat.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 15 Jun 2001         |
   ********************************************************************* */
inline Qop &Qop::operator&=(const Qop &an_operator) {
  /* append the whole operation list of the passed Qop. This will
     trigger the simplification routine in the Qop_slicelist class. */
  get_operations().append(an_operator.get_operations());
  /* return the current object */
  return *this;
}

/* *********************************************************************
   | This method is a variant of the augmentation operation for quan-  |
   | tum operators (see operator&= for further explanations). The only |
   | difference is that the list of the passed operator is moved into  |
   | the current one (not copied). Remember: this is much more effi-   |
   | cient than operator&= if we want to concatenate Qop, but the      |
   | second operator is modified (reduced to the identity).            |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 03 Oct 2001         |
   ********************************************************************* */
inline Qop &Qop::operator<<(Qop &an_operator) {
  /* splice (not copy) the whole operation list of the passed Qop. This
     will trigger the simplification routine in the Qop_slicelist class. */
  get_operations().splice(an_operator.get_operations());
  /* return the current object */
  return *this;
}

/* *********************************************************************
   | This method shifts a part of the circuit down, by calling the     |
   | corresponding operator for each slice in the operation list.      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 13 Jun 2001         |
   ********************************************************************* */
inline Qop &Qop::split(size_type head, size_type jump) {
  /* split each slice with the same parameters. */
  get_operations().split(head, jump);
  /* return the current object. */
  return *this;
}

/* *********************************************************************
   | This method inverts the order of a part of the circuit, by calling|
   | the corresponding operator for each slice in the operation list.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 13 Jun 2001         |
   ********************************************************************* */
inline Qop &Qop::invert(size_type head, size_type size) {
  /* invert each slice with the same parameters. */
  get_operations().invert(head, size);
  /* return the current object. */
  return *this;
}

/* *********************************************************************
   | This method performs a permutation of the circuit, by calling the |
   | corresponding operator for each slice in the operation list.      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      14 May 2002         |
   ********************************************************************* */
inline Qop &Qop::remap(const Qubit_list &permutation_list) {
  /* remap each slice with the same parameters. */
  get_operations().remap(permutation_list);
  /* return the current object. */
  return *this;
}

/* *********************************************************************
   | This method modifies the current operator into its adjoint: this  |
   | means inverting the order of time slices in the internal list and |
   | call the adjoin() method for each time slice. All of this is done |
   | by the adjoin() method of the private slice list.                 |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 13 Jun 2001         |
   ********************************************************************* */
inline Qop &Qop::adjoin(void) {
  /* adjoin each time slice individually + slice reversal. */
  get_operations().adjoin();
  /* return the modified operator. */
  return *this;
}

/* *********************************************************************
   | This method returns a temporary object which is a copy of the     |
   | current Qop subject up to some permutation (like invert or split, |
   | as selected by the third argument). The current Qop object is     |
   | left untouched. The behaviour is exactly that of a copy construc- |
   | tor followed by the mutable permutation.                          |
   | ----------------------------------------------------------------- |
   | The syntax for this call is not very elegant, but C++ does not    |
   | provide ternary overloadable operators (?: is ternary but cannot  |
   | be overloaded; in any case it is not intuitive, and I need two of |
   | them). Any help on simplification of this syntax will be welcome. |
   | The third argument should default to SPLIT. If the third argument |
   | is invalid, exception invalid_NM_permutation() is thrown.         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 02 Jul 2001         |
   ********************************************************************* */
inline Qop Qop::operator()(size_type arg1, size_type arg2, op_type op) const {
  /* use the third argument as operation selector. */
  switch (op) {
    /* copy the current Qop and run a split() on it. */
  case SPLIT: return Qop(*this).split(arg1, arg2); break;
    /* copy the current Qop and run an invert() on it. */
  case INVERT:  return Qop(*this).invert(arg1, arg2);  break;
    /* this default case should NEVER happen! */
  default: throw invalid_NM_permutation(); return *this; break;
  }
}

/* *********************************************************************
   | This method is a shortcut for the non mutable version of adjoin() |
   | It creates a copy of the current Qop and then adjoins the circuit |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 13 Jun 2001         |
   ********************************************************************* */
inline Qop Qop::operator!(void) const {
  return Qop(*this).adjoin();
}

/* *********************************************************************
   | This method is a shortcut for the non mutable version of offset() |
   | It creates a copy of the current Qop and then shifts the circuit  |
   | down by the required number of qubit lines. I hope this syntax is |
   | intuitive.                                                        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 02 Jul 2001         |
   ********************************************************************* */
inline Qop Qop::operator>>(size_type jump) const {
  return Qop(*this).offset(jump);
}

/* *********************************************************************
   | This method executes the operator on a given register. This means |
   | running each time slice in the associated slice list on the same  |
   | register in sequence.                                             |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 24 May 2001         |
   ********************************************************************* */
inline void Qop::operator()(const Qreg &a_register) const {
  /* let the list method do this for us */
  get_operations().operator()(a_register);
}

/* *********************************************************************
   | This method prints information about the current operator to the  |
   | supplied output stream. See the corresponding method in the slice |
   | list class for additional information.                            |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      11 Nov 2002         |
   ********************************************************************* */
inline void Qop::print_statistics(std::ostream &os) const {
  /* call the correspoding method in the slice list. */
  get_operations().print_statistics(os);
}

#endif  // __COMPOSITE_QUANTUM_OPERATORS_INLINES_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
