/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __QUANTUM_SEGMENTS_
#define __QUANTUM_SEGMENTS_

#include <iostream>                    // C++ standard I/O
#include "qinterface_types.h"          // address and size_type typedefs
#include "qexception.h"                // for "quantum" exceptions

/* *********************************************************************
   | This class holds a segment of addresses, i.e. a non empty ordered |
   | set of contiguous logical quantum bits locations; this is better  |
   | than creating a placeholder for every "allocated" quantum bit,    |
   | since unfragmented quantum registers will have a very tiny memory |
   | footprint. The most significant qubit (MSQ) is found at the seg-  |
   | ment's beginning.                                                 |
   | ----------------------------------------------------------------- |
   | Since the internal structure is quite simple, we can use the      |
   | standard copy ctor, assignement operator and dtor.                |
   | ----------------------------------------------------------------- |
   | (07 May 2001) S.Bettelli, changed internal representation from    |
   | [begin, end[ ("end" is not a valid address) to [lowest, highest]  |
   | plus a boolean variable which remembers whether the segment is    |
   | reversed or not (this allows for very fast "swap" operations!).   |
   | If the size of the segment is 1, the orientation is forced: the   |
   | reverse method does nothing in this case (so that we don't have   |
   | two representation for the same thing).                           |
   | (08 Dec 2001) S.Bettelli, changed again the internal representa-  |
   | tion to [first, last], dropped the orientation variable, which is |
   | now calculated; no more ambiguities with size-one segments. Also  |
   | added checks (with possible exceptions) to index() and operator[] |
   | (11 Dec 2001) S.Bettelli, modified operator< in order to have a   |
   | strict total order relation compatible with operator==. This      |
   | change shouldn't make any difference for non-overlapping lists.   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 17 Jan 2001         |
   | Stefano Bettelli, INFN and Trento University, 07 May 2001         |
   | Stefano Bettelli, INFN and Trento University, 08 Dec 2001         |
   ********************************************************************* */
class Qubit_segment {
public:
  // a qubit address will be an integer in the range of this unsigned type
  typedef ::qubit_address_type address;
  // a segment size will be an integer in the range of this unsigned type
  typedef ::qubit_segment_size_type size_type;
  // a class for problems with the segment size
  _Qexception(invalid_size);
  // a class for problems with out-of-range references
  _Qexception(invalid_reference);
  // a class for problems with a segment shift
  _Qexception(invalid_shift);
public:
  // the default constructor needs first (lowest) address and size
  Qubit_segment(address the_lowest = 0, size_type the_size = 1);
  // another constructor which takes the first and last absolute address
  Qubit_segment(address the_first, address the_last, bool);
public:
  // this returns the address of the first location
  address first(void) const { return first_address; }
  // this returns the address of the last location
  address last(void) const { return last_address; }
  // this returns the address of the qubit with the lowest absolute address
  address lowest(void) const { return (is_direct() ? first() : last()); }
  // this returns the address of the qubit with the highest absolute address
  address highest(void) const { return (is_direct() ? last() : first()); }
  // this is the (arbitrary) order relation
  bool operator<(const Qubit_segment &a_segment) const;
  // this is the comparison operator
  bool operator==(const Qubit_segment &a_segment) const;
  // this is the opposite of the comparison operator
  bool operator!=(const Qubit_segment &a_segment) const;
  // this returns the length of the address segment
  size_type size(void) const;
  // this calculates the index given an absolute address (may throw exception)
  size_type index(address an_address) const;
  // this calculates an offsetted qubit (may throw exception)
  address operator[](size_type offset) const;
  // this returns true if the segment contains only one address
  bool is_single(void) const { return (first() == last()); }
  // if true the addresses are registered in "direct" order
  bool is_direct(void) const { return (first() <= last()); }
  // if true the addresses are registered in reverse order
  bool is_reversed(void) const { return (! is_direct()); }
  // if true the current and the passed segments are overlapping
  bool is_overlapping(const Qubit_segment &a_segment) const;
  // return true if a join of the current and the second segment is possible
  bool can_join(const Qubit_segment &a_segment,
		Qubit_segment *save_me = NULL) const;
  // try to join the second segment into the current one
  bool try_join(const Qubit_segment &a_segment);
  // this reverses the order of locations inside the segment
  void reverse(void);
  // resize the current segment
  void resize(size_type new_size);
  // shift the current segment forward (increase addresses)
  void shift_forward(size_type a_shift);
  // shift the current segment back (decrease addresses)
  void shift_backward(size_type a_shift);
  // modify the lowest address without resizing
  void shift_lowest(address new_lowest);
private:
  // the "first" address for this segment
  address first_address;
  // the "last" address for this segment
  address last_address;
};

// this function shows the content of a segment (for debugging)
std::ostream &operator<<(std::ostream &os, const Qubit_segment &a_segment);

/* *********************************************************************
   | This method reverses the orientation of a segment; this causes    |
   | first() to be swapped with last() leaving lowest() and highest()  |
   | untouched. Size-one segments are unmodified.                      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 08 Dec 2001         |
   ********************************************************************* */
inline void Qubit_segment::reverse(void) {
  address temp  = last_address;     // save last() into temp
  last_address  = first_address;    // first() into last()
  first_address = temp;             // temp into first()
}

/* *********************************************************************
   | This is the "<" operator for segments. Segment A is said to be    |
   | "less" than segment B if its lowest address is smaller than B's   |
   | lowest address, or if they are equal and the highest address of A |
   | is smaller than the highest address of B. This is a strict total  |
   | order relation and is compatible with operator==. This relation   |
   | is very useful during sorting of lists of segments.               |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 06 Jun 2001         |
   | Stefano Bettelli, INFN and Trento University, 11 Dec 2001         |
   ********************************************************************* */
inline bool Qubit_segment::operator<(const Qubit_segment &a_segment) const {
  return (lowest() < a_segment.lowest() ||      // smaller lowest()
	  (lowest() == a_segment.lowest() &&    // equal lowest() and ...
	   highest() < a_segment.highest()));   // ... smaller highest()
}

/* *********************************************************************
   | This is the comparison operator for segments, returning true if   |
   | and only if the two segments are equal (i.e. they contain the     |
   | same addresses). In practice, the two segments must have the same |
   | first and last address in order to be equal.                      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 08 Dec 2001         |
   ********************************************************************* */
inline bool Qubit_segment::operator==(const Qubit_segment &a_segment) const {
  return (first() == a_segment.first() && last() == a_segment.last());
}

/* *********************************************************************
   | This is the opposite of the comparison operator, i.e. it returns  |
   | true if the two segments are different. It is implemented as the  |
   | negation of the comparison operator.                              |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 05 Dec 2001         |
   ********************************************************************* */
inline bool Qubit_segment::operator!=(const Qubit_segment &a_segment) const {
  return ! (*this == a_segment);
}

/* *********************************************************************
   | This method calculates the size of the current segments, i.e. the |
   | number of addresses which are contained in the segment.           |
   | ----------------------------------------------------------------- |
   | Try to optimise this routine with respect to the internal repres. |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 08 Dec 2001         |
   ********************************************************************* */
inline Qubit_segment::size_type Qubit_segment::size(void) const {
  if (is_direct()) return (last() - first() + 1);
  else             return (first() - last() + 1);
}

/* *********************************************************************
   | This method calculates the offset inside a segment (i.e. an index)|
   | once given an absolute address (if address == first() then the    |
   | calculated offset is zero); if the address is not inside the seg- |
   | ment an exception of type invalid_reference() is thrown.          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 08 Feb 2001         |
   ********************************************************************* */
inline Qubit_segment::size_type
Qubit_segment::index(address an_address) const {
  /* first case: the segment is direct and an_address is OK (which,
     in this case, means that it is less than or equal to last()). */
  if (is_direct() && an_address <= last())
    return (an_address - first());
  /* second case: the segment is reversed and an_address is OK (which,
     in this case, means that it is greater than or equal to last()). */
  else if (is_reversed() && an_address >= last())
    return (first() - an_address);
  /* third case: the reference was invalid: throw exception. */
  else throw invalid_reference();
}

/* *********************************************************************
   | This method calculates the address of the offset-th qubit (offset |
   | equal to zero means the first qubit). The passed offset must be   |
   | less than the segment size (in order to reference an address in-  |
   | side the segment) otherwise an exception of type invalid_refe-    |
   | rence() is thrown.                                                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Feb 2001         |
   ********************************************************************* */
inline Qubit_segment::address
Qubit_segment::operator[](size_type offset) const {
  if (offset < size())                // this is OK
    return (is_direct() ? (first() + offset) : (first() - offset));
  else throw invalid_reference();     // this is out-of-range
}

/* *********************************************************************
   | This method joins two segments (*this and the supplied one) if    |
   | the join is possible, leaving the result in the former without    |
   | modifying the latter. This method is useful in bookkeeping seg-   |
   | ment lists. The logic for understanding whether the join is pos-  |
   | sible or not is taken from the can_join() method. Remember to     |
   | erase the latter segment from your container if the return value  |
   | is true.                                                          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Jun 2001         |
   ********************************************************************* */
inline bool Qubit_segment::try_join(const Qubit_segment &a_segment) {
  /* try to run the joining and save the result. If the join was
     possible, the new enlarged segment is already saved in *this
     (otherwise nothing was changed). Therefore we only have to return
     the correct exit value, which is the same as that of can_join(). */
  return can_join(a_segment, this);
}

#endif  // __QUANTUM_SEGMENTS_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
