/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
/* *********************************************************************
   |                                                                   |
   |         #########        C R E D I T S        #########           |
   |                                                                   |
   | The following code was inspired by a quantum simulator written by |
   | Alexei Chepelianskii; Alexei's simulator has been used to produce |
   | the data shown in quant-ph/0202113 (available on the arXiv), now  |
   | published as: "Simulation of chaos-assisted tunneling in a semi-  |
   | classical regime on existing quantum computers", Phys. Rev. A     |
   | v.66 (2002) p.054301.                                             | 
   ********************************************************************* */
#ifndef __BASIC_QUANTUM_SIMULATOR_
#define __BASIC_QUANTUM_SIMULATOR_

#include <complex>                          // C++ complex numbers

/* *********************************************************************
   | The Qsimulator class is a proof-of-concept quantum simulator for  |
   | the Q programming language. It was written in order to show an    |
   | actual implementation of a consumer task which is fed by the      |
   | bytecode produced by the quantum library. However, it should be   |
   | stressed that in NO WAY this is the preferential simulator to be  |
   | attached to the language: better, you should use it to get inspi- |
   | ration about writing your own simulator.                          |
   | ----------------------------------------------------------------- |
   | The internals of the simulator are very simple: n qubits are ma-  |
   | naged as an array of 2^n complex numbers. The simulator interface |
   | allows for the following operations to be performed: initialisa-  |
   | tion of a qubit, measurement of a qubit (with and without wave    |
   | function collapse), Hadamard gates, phase gates and conditional   |
   | phase gates. Moreover, it is possible to ask for the probability  |
   | that a qubit is in |0> or in |1> (partial trace) and to access as |
   | constant objects the elements of the array.                       |
   | ----------------------------------------------------------------- |
   | In the array, the element indexed by the bit string "x" corres-   |
   | ponds to the projection of the quantum state on the computational |
   | basis element |x> (therefore x as an unsigned integer number must |
   | be in the range [0, 2^n - 1].                                     |
   | ----------------------------------------------------------------- |
   | (07 May 2003) S.Bettelli, changed the noise interface. The method |
   |  set_noise() can now set up both the noisy gate behaviour and the |
   | static errors behaviour (self-interaction of the quantum computer |
   | memory). For these error models you can refer to:                 |
   |                                                                   |
   |   (noisy gates) Stefano Bettelli, "A quantitative model for the   |
   |                 effective decoherence of a quantum computer with  |
   |                 imperfect unitary operations", quant-ph/0310152   |
   |   (static errors) G. Benenti, G. Casati, S. Montangero and D. L.  |
   |                 Shepelyansky, "Efficient Quantum Computing of     |
   |                 Complex Dynamics", Phys. Rev. Lett. 87, 227901    |
   |                 (2001), available as quant-ph/0107036.            |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Oct 2003         |
   ********************************************************************* */
class Qsimulator {
public:
  // the numeric type for a real number
  typedef double rnumber;
  // the numeric type for a complex number
  typedef std::complex<rnumber> cnumber;
  // an integer type for specifying addresses
  typedef unsigned long address;
  // an integer type for specifying sizes
  typedef unsigned long size_type;
public :
  // default constructor
  Qsimulator();
  // destructor (deallocating classical memory)
  ~Qsimulator();
  // index-like access to the classical memory elements
  const cnumber &operator[](address index) const { return *(data + index); }
  // this returns the number of elements in the simulator
  size_type size(void) const { return memsize; }
  // this returns the number of simulated qubits
  size_type qubits(void) const;
  // this returns the probability that the target-th qubit is |0>
  rnumber is_unset(address target) const { return 1.0 - is_set(target); }
  // this returns the probability that the target-th qubit is |1>
  rnumber is_set(address target) const;
  // this returns the amplitude of the index-th basis element
  cnumber amplitude(size_type index) const;
  // squared modulus of the amplitude of the index-th basis element
  rnumber probability(size_type index) const;
public:
  // set all the noise levels for the simulation
  void set_noise(rnumber gates, rnumber zeeman, rnumber couplings);
  // set the number of simulated qubits
  void set_numqubits(size_type num_qubits);
  // reset the memory of the simulator (perfectly)
  void reset(void);
public:
  // this initialises the target-th qubit to "value"
  void init(address target, bool value);
  // this measures the target-th qubit and returns the result
  bool measure(address target) { return do_measurement(target, true); }
  // just like measure, but the quantum state does not collapse
  bool fake_measure(address target) { return do_measurement(target, false); }
  // this applies a Hadamard gate on the target-th qubit
  void hadamard(address target);
  // this applies a phase gate on the target-th qubit
  void phase(address target, rnumber fraction);
  // this applies a conditional phase gate
  void condphase(address control, address target, rnumber fraction);
public:
  // this evolves the system with its natural hamiltonian (static errors)
  void natural_evolution(void) { do_zeeman_terms(); do_coupling_terms(); }
  // this chooses new values for the natural hamiltonian coefficients
  void regenerate_staterr(void) { generate_zeeman(zeeman_noise);
                                  generate_couplings(coupling_noise); }
  // (re)initialisation of the random number generator
  unsigned long init_random_seed(unsigned long seed = 0);
private:
  // coefficients for the Hadamard gate
  void prepare_hadamard(cnumber &m00, cnumber &m01);
  // coefficient for the phase or conditional phase gate
  cnumber prepare_phase(rnumber fraction);
  // reimplementation of norm() of complex.h for performance
  static rnumber norm2(const cnumber &z);
  // this implements a measurement (with or without collapse)
  bool do_measurement(address target, bool collapse);
  // this collapses the quantum state of the target-th qubit to value
  void do_collapse(address target, bool value, rnumber probability);
  // setup the Zeeman terms for the static interactions
  void generate_zeeman(rnumber noise);
  // setup the coupling terms for the static interactions
  void generate_couplings(rnumber noise);
  // apply the natural evolution given by Zeeman terms
  void do_zeeman_terms(void);
  // apply the natural evolution given by coupling terms
  void do_coupling_terms(void);
  // apply one perfect Hadamard transformation to each qubit
  void do_multiHadamard(void);
private:
  // intensity of errors in noisy gates
  rnumber gate_noise;
  // intensity of static errors (Zeeman terms)
  rnumber zeeman_noise;
  // intensity of static errors (coupling terms)
  rnumber coupling_noise;
  // this is the amount of classical memory we use
  size_type memsize;
  // this is a pointer to the memory block
  cnumber *data;
  // a pointer to the chosen values for the Zeeman terms
  cnumber *zeeman_terms;
  // a pointer to the chosen values for the coupling terms
  cnumber *coupling_terms;
};

/* *********************************************************************
   | For the time being, the simulator is a global object. It is de-   |
   | clared as an external object here, so that each file including    |
   | the class declaration will have access to the simulator too.      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Nov 2002         |
   ********************************************************************* */
extern Qsimulator simulator;

#endif  // __BASIC_QUANTUM_SIMULATOR_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
