/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __QUANTUM_SUMMARIES_
#define __QUANTUM_SUMMARIES_

#include <iostream>                              // C++ standard I/O
#include "qsegment.h"                            // base class declaration

/* *********************************************************************
   | This class is an extension to the Qubit_segment class containing  |
   | four additional fields:                                           |
   | 1)    the usage count for the segment.                            |
   | 2/3)  the usage counts for the borders of the segment.            |
   | 4)    A pointer to an eterogeneous data area for keeping          |
   |       additional information (a cookie).                          |
   | ----------------------------------------------------------------- |
   | The usage count is used, e.g. in the address manager main list,   |
   | which must remember how many registers are currently using a      |
   | given segment. The usage count may of course overflow, an excep-  |
   | tion will notify it (in a few years). This kind of Qubit_segment  |
   | is considered free if the usage count is zero (obvious).          |
   | The usage counts for the borders of the segment are analogous.    |
   | They are used to remember how many registers have the given       |
   | segment border as an actual segment border in their internal      |
   | segment list. These data are necessary if we want to run a recol- |
   | lection strategy in the address manager AND be sure that the      |
   | "union" assumption (i.e. that each segment in a register is union |
   | of segments in the main list of the address manager) holds.       |
   | The "cookie" is stored using a pointer to a Qubit_cookie object.  |
   | Indeed the Qubit_cookie class is empty, but any user can define   |
   | other classes inheriting from it which will be resolved at run    |
   | time through their virtual table. This mechanism will allow us to |
   | store information of a type which is still not known (see the     |
   | Qaddress_manager class ....)                                      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 20 Jan 2001         |
   | Stefano Bettelli, INFN and Trento University, 31 Jan 2001 (cookie)|
   | Stefano Bettelli, INFN and Trento University, 23 Apr 2001 (border)|
   ********************************************************************* */
class Qubit_cookie { };                          // empty cookie class ...
class Qubit_summary : public Qubit_segment {
public:
  // this is the type for the counter field
  typedef unsigned short counter_type;
public:
  // base class constructor (first and size)
  Qubit_summary(address the_first = 0, size_type the_size = 1);
  // virtual costructor (it must be run by all inheriting classes!)
  virtual ~Qubit_summary();
public:
  // increment the segment usage count
  void add_bulk(void) { ++count; }
  // increment the minimum address border usage count
  void add_left_border(void) { ++left_count; }
  // increment the maximum address border usage count
  void add_right_border(void) { ++right_count; }
  // increment all the usage counts together
  void add(void)
  { add_bulk(); add_left_border(); add_right_border(); }
  // decrement the segment usage count
  void del_bulk(void) { --count; }
  // decrement the minimum address border usage count
  void del_left_border(void) { --left_count; }
  // decrement the maximum address border usage count
  void del_right_border(void) { --right_count; }
  // decrement all the usage counts together
  void del(void)
  { del_bulk(); del_left_border(); del_right_border(); }
  // get the segment usage count for this segment
  counter_type get_bulk(void) const { return count; }
  // get the minimum address border usage count
  counter_type get_left_border(void) const { return left_count; }
  // get the maximum address border usage count
  counter_type get_right_border(void) const { return right_count; }
  // test the segment for not being busy (it uses segment usage count)
  bool is_free(void) const { return (get_bulk() == 0); }
  // set the segment usage count for this segment
  void set_bulk(counter_type a_count) { count = a_count; }
  // set the minimum address border usage count for this segment
  void set_left_border(counter_type a_count) { left_count = a_count; }
  // set the maximum address border usage count for this segment
  void set_right_border(counter_type a_count) { right_count = a_count;}
  // set all the usage counts together
  void set(counter_type bulk_count,
	   counter_type left_count,
	   counter_type right_count);
  // get the cookie pointer stored inside the object
  Qubit_cookie *get_cookie(void) { return cookie; }
  // get the cookie pointer stored inside the object (const version)
  const Qubit_cookie *get_cookie(void) const { return cookie; }
  // save a new cookie pointer into the object
  void set_cookie(Qubit_cookie *new_cookie);
private:
  // this field holds the usage count for this segment
  counter_type count;
  // the usage count for the least absolute address border
  counter_type left_count;
  // the usage count for the maximum absolute address border
  counter_type right_count;
  // this fiels holds the user cookie (generic)
  Qubit_cookie *cookie;
  // private method to release the cookie pointer
  void release_cookie(void);
};

// this function shows the content of a summary (for debugging)
std::ostream &operator<<(std::ostream &os, const Qubit_summary &a_summary);
// this function shows the content of a cookie (placeholder)
std::ostream &operator<<(std::ostream &os, const Qubit_cookie &a_cookie);

/* *********************************************************************
   | The destructor is declared virtual, because it must be called by  |
   | all inheriting classes. It does nothing more than releasing the   |
   | cookie, that is it deallocates the object pointed to by the cookie|
   | pointer (the real derived object is indeed destroyed, thanks to   |
   | the virtual table, at least so I think). This policy means that   |
   | the cookie must not be shared with other object ...               |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Jan 2001         |
   ********************************************************************* */
inline Qubit_summary::~Qubit_summary() {
  release_cookie();
}

/* *********************************************************************
   | This method saves a new cookie pointer into the object.           |
   | If however a cookie is already set, it is deallocated first.      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Jan 2001         |
   ********************************************************************* */
inline void Qubit_summary::set_cookie(Qubit_cookie *new_cookie) {
  release_cookie();
  cookie = new_cookie;
}

/* *********************************************************************
   | This (private) method takes care to deallocate the object pointed |
   | to by the cookie pointer. Since in general this pointer will be   |
   | related to an object of a class inheriting from Qubit_cookie, the |
   | virtual table will provide information for deleting the most      |
   | derived object at run time. The pointer is then set to NULL.      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Jan 2001         |
   ********************************************************************* */
inline void Qubit_summary::release_cookie(void) {
  if (cookie) delete cookie;
  cookie = NULL;
}

#endif  // __QUANTUM_SUMMARIES_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
