/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#ifndef __BITSET_FOR_QUANTUM_REGISTER_INITIALISATION_
#define __BITSET_FOR_QUANTUM_REGISTER_INITIALISATION_

#include <iostream>                         // for ouputs streams
#include "qexception.h"                     // for quantum exceptions
#include "qinterface_types.h"               // for quantum_message_type

/* *********************************************************************
   | This class is a container for boolean values, needed for initia-  |
   | lisation and measurement of quantum registers with arbitrary size |
   | (a replacement for integers with an arbitrary number of bits).    |
   | It packs the boolean values in objects of type storage_type,      |
   | which is the same as the ::quantum_message_type, so that interac- |
   | tion with the command/message queue of the quantum interface will |
   | very easy and fast. The first bit in this container is the MSB.   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Sep 2001         |
   ********************************************************************* */
class Qbitset {
public:
  // the type used for a unit of storage
  typedef ::quantum_message_type storage_type;
  // a type for counting boolean values
  typedef unsigned long size_type;
  // the size in bits of one unit of storage
  static const size_type storage_type_size = QUANTUM_MESSAGE_TYPE_SIZE;
public:
  // a class for problems with number of bits
  _Qexception(qbitset_overflow);
  // a class for out-of-boundary accesses
  _Qexception(qbitset_out_of_boundaries);
public:
  // inhibit default constructor
  Qbitset();
  // a constructor with number of bits and value
  Qbitset(storage_type a_size, size_type a_value = 0);
  // copy constructor for a bitset
  Qbitset(const Qbitset &a_bitset);
  // assignement operator for a bitset
  Qbitset &operator=(const Qbitset &a_bitset);
  // non virtual destructor (release storage elements)
  ~Qbitset() { if (data) delete [] data; data = NULL; }
public:
  // this method sets the index-th bit in the collection
  void set(size_type index, bool value = true);
  // this method reads the index-th bit in the collection
  bool read(size_type index) const;
  // this method sets the index-th storage element in the collection
  void set_element(size_type index, storage_type value);
  // this method reads the index-th storage element in the collection
  storage_type get_element(size_type index) const;
  // this method returns true if the bitset contains no bits
  bool empty(void) const { return (real_size == 0); }
  // this method returns the number of bits stored in the bitset
  size_type size(void) const { return real_size; }
  // this method returns the number of elements stored in the bitset
  size_type elements(void) const { return num_elements; }
  // cast operator to a quantum_message_type object (if possible)
  operator ::quantum_message_type(void) const { return to_integer_value(); }
private:
  // this calculates the number of bits necessary for storing a value
  size_type bits_for(storage_type a_value);
  // this calculates the number of storage units for holding "a_size" bits.
  size_type calculate_container_size(size_type a_size);
  // this method returns the integer conversion of the bitset (if possible)
  const storage_type to_integer_value(void) const;
  // this is a mask to the MSB in each data element
  static const storage_type MSB_mask =storage_type(0x1<<(storage_type_size-1));
  // this method returns the elements which contains the index-th bit
  size_type index_to_element(size_type index) const;
  // a bitmask selecting the index-th bit within its storage element
  size_type index_to_mask(size_type index) const;
  // this method duplicates a data area with "num_elements" elements
  void duplicate_data(const storage_type *data_source);
private:
  // number of bits in the bitset (<= number of bits in the data area)
  size_type real_size;
  // number of storage elements
  size_type num_elements;
  // the packed bits (in a container of storage_type objects)
  storage_type *data;
};

// this output function shows the content of a bitset (for debugging)
std::ostream &operator<<(std::ostream &os, const Qbitset &a_bitset);

/* *********************************************************************
   | This is a helper function which calculates how many units of      |
   | storage are necessary to hold "a_size" boolean values. It uses    |
   | the constant storage_type_size which is the number of bits in a   |
   | storage_type object.                                              |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Sep 2001         |
   ********************************************************************* */
inline Qbitset::size_type Qbitset::calculate_container_size(size_type a_size) {
  return (a_size - 1 + storage_type_size) / storage_type_size;
}

/* *********************************************************************
   | This is the implementation of the copy constructor, which simply  |
   | copies the internal counters and duplicates the data area.        |
   | (Remember to set the data pointer to NULL, since this is not done |
   | by all compilers).
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 02 Jan 2002         |
   ********************************************************************* */
inline Qbitset::Qbitset(const Qbitset &a_bitset) :
  real_size(a_bitset.real_size),
  num_elements(a_bitset.num_elements),
  data(NULL)
{ duplicate_data(a_bitset.data); }

/* *********************************************************************
   | This method returns the storage element containing the index-th   |
   | bit. As usual index=0 is the MSB; the calculation is very simple  |
   | since it can rely on integer division.                            |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      20 Jul 2002         |
   ********************************************************************* */
inline Qbitset::size_type Qbitset::index_to_element(size_type index) const {
  return (index / storage_type_size);
}

/* *********************************************************************
   | This method returns a bitmask which selects the index-th bit      |
   | within its storage element (it is thus to be used in conjunction  |
   | with "index_to_element"). As usual index=0 is the MSB; the cal-   |
   | culation is very simple since it amounts to a shift of MSB_mask.  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      20 Jul 2002         |
   ********************************************************************* */
inline Qbitset::size_type Qbitset::index_to_mask(size_type index) const {
  return (MSB_mask >> (index % storage_type_size));
}

/* *********************************************************************
   | This method sets the index-th storage element to the specified    |
   | value. Out of bound. accesses are catched (an exception is thrown)|
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      20 Jul 2002         |
   ********************************************************************* */
inline void Qbitset::set_element(size_type index, storage_type value) {
  /* catch out-of-boundary accesses. size_type should be unsigned,
     so we must only check for overflows. */
  if (index >= elements()) throw qbitset_out_of_boundaries();
  /* store the new value in the storage element */
  data[index] = value;
}

/* *********************************************************************
   | This method reads the index-th storage element. Out of boundaries |
   | accesses are catched (an exception is thrown).                    |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      20 Jul 2002         |
   ********************************************************************* */
inline Qbitset::storage_type Qbitset::get_element(size_type index) const {
  /* catch out-of-boundary accesses. size_type should be unsigned,
     so we must only check for overflows. */
  if (index >= elements()) throw qbitset_out_of_boundaries();
  /* return the value of the index-th element */
  return data[index];
}

/* *********************************************************************
   | This method sets the index-th bit in this bit container to the    |
   | passed value (the default is "true"). As usual, index=0 corres-   |
   | ponds to the MSB and index=size()-1 corresponds to the LSB. Out   |
   | of boundaries accesses are catched (an exception is thrown).      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      20 Jul 2002         |
   ********************************************************************* */
inline void Qbitset::set(size_type index, bool value) {
  /* catch out-of-boundary accesses. size_type should be unsigned,
     so we must only check for overflows. */
  if (index >= size()) throw qbitset_out_of_boundaries();
  /* if the value is to be set, OR the element with the mask */
  if (value)
    data[index_to_element(index)] |=  index_to_mask(index);
  /* if the value is to be unset, AND the element with the negated mask */
  else
    data[index_to_element(index)] &= ~index_to_mask(index);
}

/* *********************************************************************
   | This method reads the index-th bit in this bit container. As usu- |
   | al, index=0 corresponds to the MSB and index=size()-1 corresponds |
   | to the LSB. Out of boundaries accesses are catched (an exception  |
   | is thrown).                                                       |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      20 Jul 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      27 Oct 2003         |
   ********************************************************************* */
inline bool Qbitset::read(size_type index) const {
  /* catch out-of-boundary accesses. size_type should be unsigned,
     so we must only check for overflows. */
  if (index >= size()) throw qbitset_out_of_boundaries();
  /* return the value of the index-th bit. Explicitely compare
     the result of the AND with zero for portability. */
  return 0 != (data[index_to_element(index)] & index_to_mask(index));
}

#endif  // __BITSET_FOR_QUANTUM_REGISTER_INITIALISATION_
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***

