/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <vector>                        // STL vectors
#include <cmath>                         // for the ldexp routine
#include "qinterface.h"                  // class definition
#include "qop_codes.h"                   // constants for operation codes
#include "qsimulator.h"                  // access to the quantum simulator

/* *********************************************************************
   | This routine is an example of decoder for the bytecode buffer. It |
   | swaps the buffer into a local structure and then consumes it. It  |
   | is assumed here that the buffer at the beginning contains only    |
   | complete initialisations, measurements, time slice operations or  |
   | control commands for the quantum device.                          |
   | In each cycle of the main loop a bytecode element is read (the    |
   | operation code) and its value is used to process the follosing    |
   | elements. Every unrecognisable code gets printed to stderr.       |
   | ----------------------------------------------------------------- |
   | (8 sep 2002) S.Bettelli Don't call POP twice in the same function |
   | call! The call order is undefined in this case! I'm going to pro- |
   | provide a solution for this in the form of a "define" statement.  |
   | ----------------------------------------------------------------- |
   | (02 Apr 2003) S.Bettelli. Just to make this library a little sim- |
   |  pler to manage for me, I have decided to add some code for a toy |
   |  quantum simulator. This routine has been modified in order to    |
   |  link to this simulator; it is to be pointed out, however, that   |
   |  this simulator is in no way the preferred one and that in order  |
   |  to link to something else, only the routine(s) in this file      |
   |  should be rewritten (you can use the methods as templates, only  |
   |  the actual calls to the simulator routies should be touched).    |
   | ----------------------------------------------------------------- |
   | (07 May 2003) S.Bettelli. Each time the simulation of a timeslice |
   |  is over, we make a call to the method "natural_evolution" of our |
   |  simulator, so that the environmental decoherence and the self    |
   |  interaction of the quantum computer memory can make some damage. |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      14 May 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      19 Jul 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      02 Apr 2003         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      07 May 2003         |
   ********************************************************************* */
void Qinterface::run_evolution(void) {
  /* If we are running a threaded run_evolution(), the safest approach
     is to move all the code from the bytecode buffer to a secondary buffer
     which will be then consumed without interference wiht the bytecode
     producer. Of course, the buffer is marked as busy during the bytecode
     swapping. If everything is serial, the following lines are actually
     unuseful, but we don't care since swapping is very fast. */
  command_queue_type queue_copy;
  channel_busy();
  queue_copy.swap(command_queue);
  channel_free();
  /* the POP macro is a shortcut for getting a value out of a bytecode
     buffer and delete it at the same time from the buffer. The READ_SEGMENT
     macro solves a problem of an undefined call sequence (POP should be
     called twice in the argument list of the Qubit_segment constructor).
     The last macro turns on and off the output to stdout (our "simulator") */
#define POP read_queue(queue_copy)
#define READ_SEGMENT(s) Qubit_segment::address first_##s = POP;  \
                        Qubit_segment::address last_##s  = POP;  \
			Qubit_segment s(first_##s, last_##s, true)
  /* Now enter the real "simulation" loop. Each cycle will consume a
     complete time slice or an initialisation or a measurement or a
     control for the quantum device driver. This is safe since
     Qinterface_base blocks access to the bytecode buffer while
     it is writing one complete block of information. */
  while (! queue_copy.empty()) {
    /* pop the first element from the bytecode buffer. It should be a
       command, which we use to decide about further processing. */
    message_type command_code = POP;
    switch (command_code) {
      /* ************************************************************
	 | INITIALISATION OF A QUANTUM REGISTER                     |
	 ************************************************************ */
    case OPCODE_INITIALISATION: {
      /* the first value in the buffer now is the number of bits to be
	 initialised. Build a Qbitset with the appropriate size. */
      Qbitset init(POP);
      /* for each message_type object containing boolean values, read
	 the object and copy it into the "init" container. */
      for (message_type j = 0; j < init.elements(); ++j)
	init.set_element(j, POP);
      /* the next value is the number of [first,last] pairs
	 which will follow, i.e. the number of segments. */
      message_type reg_segments = POP;
      /* Read the data for the segments and initialise them. */
      message_type bit_counter = 0;
      for (message_type j = 0; j < reg_segments; ++j) {
	READ_SEGMENT(segment);
	for (message_type k = 0; k < segment.size(); ++k, ++bit_counter)
	  /* ask the simulator to perform the initialisation. */
	  simulator.init(segment[k], init.read(bit_counter));
      }
    } break;
    /* ************************************************************
       | MEASUREMENT OF A QUANTUM REGISTER                        |
       ************************************************************ */
    case OPCODE_MEASURE: {
      /* the next value is the number of [first,last] pairs
	 which will follow, i.e. the number of segments. */
      message_type reg_segments = POP;
      /* Save all the segments for later use. Also calculate the total
	 number of qubits to be measured. Use an STL vector for this,
	 since reg_segments is a variable. */
      std::vector<Qubit_segment> all_segments(reg_segments);
      message_type num_bits = 0;
      for (message_type j = 0; j < reg_segments; ++j) {
	READ_SEGMENT(segment);
	num_bits += (all_segments[j] = segment).size();
      }
      /* Read the qubit addresses and "measure" them (asking the simulator
	 to measure and return the results), immediately saving the result
	 in a qubit-set. Increment bit_index only in the inner loop! */
      Qbitset results(num_bits);
      message_type bit_index = 0;
      for (message_type j = 0; j < reg_segments; ++j)
	for (message_type k = 0; k < all_segments[j].size(); ++k, ++bit_index)
	  results.set(bit_index, simulator.measure(all_segments[j][k]));
      /* write the data to the feedback buffer. */
      channel_feed_busy();
      write_feedback(results.size());
      for (message_type j = 0; j < results.elements(); ++j)
	write_feedback(results.get_element(j));
      channel_feed_free();
    } break;
    /* ************************************************************
       | PHYSICAL LOCATION OF A QUANTUM REGISTER                  |
       ************************************************************ */
    case OPCODE_LOCATE: {
      /* the next value is the number of [first,last] pairs
	 which will follow, i.e. the number of segments. */
      message_type reg_segments = POP;
      /* Then, read the data for the registers and "locate" them. For the
	 time being, this task is very simple since we only have to copy
	 the request into the feedback channel (i.e. a trivial device
	 does not apply any further mapping to qubits). You should anyway
	 REPLACE THE FOLLOWING PIECE OF CODE with a real location routine
	 if needed (for instance, if the simulator performs some additional
	 qubit relocation). */
      channel_feed_busy();
      write_feedback(reg_segments);
      for (message_type j = 0; j < reg_segments; ++j) {
	READ_SEGMENT(segment);
	write_feedback(segment.first());
	write_feedback(segment.last());
      }
      channel_feed_free();
    } break;
    /* ************************************************************
       | TENSOR PRODUCT OF HADAMARD GATES                         |
       ************************************************************ */
    case OPCODE_HADAMARD: {
      /* The Hadamard gate requires just one list of registers, so its
	 implementation is very similar to the procedure for measurements. */
      message_type reg_segments = POP;
      for (message_type j = 0; j < reg_segments; ++j) {
	READ_SEGMENT(segment);
	for (message_type k = 0; k < segment.size(); ++k)
	  /* ask the simulator to perform the Hadamard gate on segment[k]. */
	  simulator.hadamard(segment[k]);
      }
    } break;
    /* ************************************************************
       | TENSOR PRODUCT OF PHASE GATES                            |
       ************************************************************ */
    case OPCODE_PHASE: {
      /* The phase gate "simulation" works exactly like that of the
	 Hadamard gate, but we need to read and decode an additional
	 element from the buffer at the beginning: the phase parameter.
	 See the Qop_phase_like class for details on the encoding. */
      message_type parameter = POP;
      message_type reg_segments = POP;
      /* an explicit cast is necessary to disambiguate ldexp. */
      double phase = ldexp(static_cast<double>(parameter),
			   - QUANTUM_MESSAGE_TYPE_SIZE);
      for (message_type j = 0; j < reg_segments; ++j) {
	READ_SEGMENT(segment);
	for (message_type k = 0; k < segment.size(); ++k) {
	  /* ask the simulator to perform the phase gate on segment[k]. */
	  simulator.phase(segment[k], phase);
	}
      }
    } break;
      /* ************************************************************
	 | TENSOR PRODUCT OF CONDITIONAL PHASE GATES                |
	 ************************************************************ */
    case OPCODE_COND_PHASE: {
      /* Conditional phase gates have and additional parameter, like
	 simple phase gates, and they also need two lists of registers
	 (we can find corresponding registers in pairs). */
      message_type parameter = POP;
      message_type twice_reg_segments = POP;
      /* an explicit cast is necessary to disambiguate ldexp. */
      double phase = ldexp(static_cast<double>(parameter),
			   - QUANTUM_MESSAGE_TYPE_SIZE);
      for (message_type j = 0; j < twice_reg_segments/2; ++j) {
	READ_SEGMENT(segment_control);
	READ_SEGMENT(segment_target);
	for (message_type k = 0; k < segment_control.size(); ++k)
	  /* ask the simulator to perform the controlled phase gate. */
	  simulator.condphase(segment_control[k], segment_target[k], phase);
      }
    } break;
    /* ************************************************************
       | YOU SHOULD'T REACH THIS POINT!                           |
       ************************************************************ */
    case OPCODE_SWAP: 
      std::cerr << "This code should not be sent!\n"; exit(-1);
    default:
      std::cerr << "I should not reach this point!\n"; exit(-1);
    }
    /* This is the end of the simulation of one time slice. If our quantum
       computer is parallel (we assume this!), there is now and only now
       a pause during which the natural evolution of the quantum memory
       possibly introduces errors. Call the appropriate method for the
       simulator, unless the command was a control command. */
    if (command_code != OPCODE_LOCATE) simulator.natural_evolution();
  }

}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
