/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <algorithm>                     // for STL sort & adjacent_find
#include "qlist.h"                       // base class declaration

/* *********************************************************************
   | Another constructor of a list which accepts an STL vector filled  |
   | with qubit addresses. These addresses are transformed into seg-   |
   | ments and pushed into the list. There is a check that all these   |
   | addresses are distinct, since a Qubit_list must be overlap free;  |
   | If the vector is not overlap free, this ctor throws an exception  |
   | of type invalid_constructor(). The list is then minimised.        |
   | ----------------------------------------------------------------- |
   | The test is performed, by copying the user's vector into a local  |
   | object, sorting it and testing it for the presence of adjacent    |
   | identical elements. The test is optimised for vectors with only   |
   | one or two elements; we don't need a copy in this case (do I have |
   | to optimise also for other cases?                                 |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      14 May 2002         |
   ********************************************************************* */
Qubit_list::Qubit_list(const std::vector<address> &a_list) {
  /* a shortcut for an STL vector of qubit addresses. */
  typedef std::vector<address> address_vector;
  /* get the size of the passed vector */
  address_vector::size_type list_size = a_list.size();
  /* if size is equal to one (or zero), we cannot have problems. But if
     size is larger than one, we must explicitely check that all the
     elements in the passed list are distinct. */
  if (list_size > 1) {
    /* if size is equal to two, the check is strightforward. */
    if (list_size == 2)
      { if (a_list.front() == a_list.back()) throw invalid_construction(); }
    /* in the general case we go through the complicated thing. */
    else {
      /* copy the user's vector into a modifiable local vector. */
      address_vector list_copy(a_list);
      /* sort this vector, so that addresses with the same value will be
	 adjacent. Then test if any of these address groups is present
	 (adjacent_find will do it for us): in this case an exception of
	 type "invalid_construction" must be thrown. Use std::sort and
	 not sort alone, because Qubit_list derives from an STL list and
	 STL lists have a sort method! */
      std::sort(list_copy.begin(), list_copy.end());
      if (adjacent_find(list_copy.begin(), list_copy.end()) != list_copy.end())
	throw invalid_construction();
    }
  }
  /* if we get to this point, we are sure that all the elements in the
     user's vector are distinct. Now, transform each element into a qubit
     segment and push it into the list, without further investigations. */
  for (address_vector::const_iterator element = a_list.begin();
       element != a_list.end(); ++element)
    base_class::push_back(Qubit_segment(*element, 1));
  /* be sure that the list is reduced to minimality. */
  make_minimal();
}

/* *********************************************************************
   | The size() method returns the number of qubits addressed inside   |
   | the list; this is NOT the size of the list (i.e. the number of    |
   | segments) because each segment can hold more than one address.    |
   | What we have to do is to iterate over the list and accumulate the |
   | sizes of the single segments. Optimised for lists of one or zero  |
   | segments.                                                         |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 17 Jan 2001         |
   ********************************************************************* */
Qubit_list::size_type Qubit_list::size(void) const {
  /* if the list has only one segment, return its size. */
  if (is_single()) return front().size();
  /* if the list is empty, return 0 */
  if (is_empty()) return 0;
  /* do it the complicated way, reset the number of qubits. */
  size_type list_size = 0;
  /* accumulate all qubits of this segment. */
  for (const_iterator segment = begin(); segment != end(); ++segment)
    list_size += segment->size();
  /* return the accumulated number of qubits. */
  return list_size;
}

/* *********************************************************************
   | This method returns the highest address in the segments of the    |
   | current list. What we have to do is to iterate over the list and  |
   | call the highest() method for each individual segment. Optimised  |
   | for lists with exactly one segment. If the list is empty, an      |
   | exception of type invalid_limit is thrown.                        |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Oct 2002         |
   ********************************************************************* */
Qubit_list::address Qubit_list::highest(void) const {
  /* if there is only one segment, return its highest value. */
  if (is_single()) return front().highest();
  /* if there are no segments, issue an exception. */
  if (is_empty()) throw invalid_limit();
  /* do it the complicated way, reset the highest address sofar */
  address highest_address = 0;
  /* check highest address in each segment */
  for (const_iterator segment = begin(); segment != end(); ++segment)
    if (segment->highest() > highest_address)
      highest_address = segment->highest();
  /* return the highest address found. */
  return highest_address;
}

/* *********************************************************************
   | This method returns the lowest address in the segments of the     |
   | current list. What we have to do is to iterate over the list and  |
   | call the lowest() method for each individual segment. Optimised   |
   | for lists with exactly one segment. If the list is empty, an      |
   | exception of type invalid_limit is thrown.                        |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Oct 2002         |
   ********************************************************************* */
Qubit_list::address Qubit_list::lowest(void) const {
  /* if there is only one segment, return its lowest value. */
  if (is_single()) return front().lowest();
  /* if there are no segments, issue an exception. */
  if (is_empty()) throw invalid_limit();
  /* do it the complicated way, reset the lowest address sofar */
  address lowest_address = 0;
  /* check lowest address in each segment */
  for (const_iterator segment = begin(); segment != end(); ++segment)
    if (segment->lowest() < lowest_address)
      lowest_address = segment->lowest();
  /* return the lowest address found. */
  return lowest_address;
}

/* *********************************************************************
   | This method implements the reversal of the natural address orde-  |
   | ring inside a list of addresses. This function is linear time in  |
   | the number of segments. The return value is the object itself.    |
   | (03 Dec 2001) S.Bettelli: changed return type from void.          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 27 Jul 2001         |
   ********************************************************************* */
Qubit_list &Qubit_list::reverse(void) {
  /* reverse the order of segments inside the list (without copies, the
     STL list member function only modifies the prev/next pointers); this
     operation is linear time and all iterators are still valid and they
     point to the same elements. */
  base_class::reverse();
  /* now reverse each segment separately. The composition of this
     operation and the previous one is equivalent to a reversal of
     the overall sequence of qubits in this list. */
  for (iterator a_segment = begin(); a_segment != end(); ++a_segment)
    a_segment->reverse();
  /* return the current object */
  return *this;
}

/* *********************************************************************
   | This method checks whether any pair of consecutive segments in    |
   | the current list cannot be joined (i.e. whether the list is mini- |
   | mal or not). The return value is "true" if the list is minimal,   |
   | "false" otherwise. The routine works by trying each pair of con-  |
   | secutive segments, thus its complexity is linear in the number of |
   | segments inside the list. Optimised for lists of 0 or 1 segment.  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Jun 2001         |
   ********************************************************************* */
bool Qubit_list::is_minimal(void) const {
  /* don't be fooled by empty lists, return immediately. Also,
     if the list has only one segment, it is already minimal! */
  if (! is_multiple()) return true;
  /* have two iterators in the list. Let the first point to the
     beginning of the list and the second to the following element. */
  const_iterator first_segment = begin();
  const_iterator second_segment = first_segment;
  ++second_segment;
  /* loop until the second iterator reaches the end of the list. */
  while (second_segment != end()) {
    /* return false if we found a possible join */
    if (first_segment->can_join(*second_segment)) return false;
    /* otherwise increase both iterators. */
    ++first_segment;
    ++second_segment;
  }
  /* if you get here, the list is minimal. Return true. */
  return true;
}

/* *********************************************************************
   | This method joins those segments which can be joined. Therefore,  |
   | after a call to this method the list is minimal by sure (i.e. if  |
   | you ask is_minimal() the answer is always true). Optimised for    |
   | lists with zero or one segments.                                  |
   | ----------------------------------------------------------------- |
   | (04 Dec 2001) S.Bettelli, bug corrected. If a join is successful, |
   |   the iterator must be tried again, not skipped!                  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Jun 2001         |
   ********************************************************************* */
void Qubit_list::make_minimal(void) {
  /* skip lists with zero or one segments. */
  if (! is_multiple()) return;
  /* Iterate through the list and join as many segments as possible. */
  iterator a_segment = begin();
  while (a_segment != end()) {
    /* Try to join the current segments with its follower. */
    bool success = try_join_and_cleanup_forward(a_segment);
    /* move on if the join was unsuccessful; if it was, try
       again with the same iterator. */
    if (! success) ++a_segment;
  }
}

/* *********************************************************************
   | This method checks whether there is any overlap between the       |
   | current list and the segment passed as argument. It returns       |
   | "false" if no overlap is found, "true" otherwise.                 |
   | ----------------------------------------------------------------- |
   | There is an optimisation when the current list contains only zero |
   | (can this happen?) or one (very typical) segments.                |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      11 Nov 2002         |
   ********************************************************************* */
bool Qubit_list::is_overlapping(const Qubit_segment &a_segment) const {
  /* return immediately if the current list is empty. */
  if (this->is_empty()) return false;
  /* if the current list contains only one segment (very typical)
     simply return the result of the comparison. */
  if (this->is_single())
    return (this->front()).is_overlapping(a_segment);
  /* things are not so easy then ... loop through the list of segments
     and test each segment separately against "a_segment". Return as
     soon as a test is positive. */
  for (const_iterator current_segment = begin();
       current_segment != end(); ++current_segment)
    if (current_segment->is_overlapping(a_segment)) return true;
  /* if you get here, all the tests failed. Return "false" */
  return false;
}

/* *********************************************************************
   | This method reorders all the addresses in the list in increasing  |
   | value order. Since the list is overlap free, this means reorde-   |
   | ring all the segments with respect to operator<. After this, all  |
   | segments are forced to the "direct" state and the final list is   |
   | reduced to minimality. Optimised for lists with 0 or 1 segments.  |
   | ----------------------------------------------------------------- |
   | Be aware that calling this routine can be not what you want to do |
   | on your list: sometimes the order is meaningful!                  |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      26 Oct 2002         |
   ********************************************************************* */
void Qubit_list::reorder(void) {
  /* If the list contains only zero segments, return immediately;
     if there is only one segment, force it to "direct" and return. */
  if (is_single() && begin()->is_reversed()) begin()->reverse();
  if (! is_multiple()) return;
  /* Use STL sort, since we already have operator< for segments. */
  base_class::sort();
  /* Be sure all segments are in direct order. */
  for (iterator a_segment = begin(); a_segment != end(); ++a_segment)
    if (a_segment->is_reversed()) a_segment->reverse();
  /* Run a minimisation routine on the current list */
  make_minimal();
}

/* *********************************************************************
   | This method takes two lists (the first is implicitely the current |
   | list), reorders them in increasing value order and eliminates     |
   | from each list the common addresses. It is possible that, after   |
   | this, one or both lists are empty. Note however that the lists    |
   | are reordered even if no annihilation is possible.                |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      28 Oct 2002         |
   ********************************************************************* */
void Qubit_list::reorder_and_annihilate(Qubit_list &second_list) {
  /* take a reference to the current list. */
  Qubit_list &first_list = *this;
  /* reorder both lists (optimisations for 0 and 1 segment lists) */
  first_list.reorder(); second_list.reorder();
  /* if any of the lists is empty, return immediately. */
  if (first_list.is_empty() || second_list.is_empty()) return;
  /* iterate through the two lists, till we reach the end of one of them. */
  iterator first_segment  = first_list.begin();
  iterator second_segment = second_list.begin();
  while (first_segment  != first_list.end() &&
	 second_segment != second_list.end()) {
    /* deal with the segments so that they are disjoint or identical. */
    first_list.break_overlap_with(second_list, first_segment, second_segment);
    /* if the two segments are now identical, eliminate both. Save
       the return value as a new starting point for the iterators. */
    if (*first_segment == *second_segment) {
      first_segment  = first_list.erase(first_segment);
      second_segment = second_list.erase(second_segment);
    } else {
      /* otherwise, increase the iterator to the segment
	 with the smallest lowest value. */
      if (*first_segment < *second_segment)
	++first_segment; else ++second_segment;
    }
  }
}

/* *********************************************************************
   | This method compares two direct segments from different lists,    |
   | and, if they are different and overlapping, it applies a segment  |
   | split in order to eliminate the overlap. Therefore, after a call  |
   | to this method, the segments pointed to by "first_segment" and    |
   | "second_segment" are identical or non overlapping. If the lists   |
   | were not distinct or the segments were not direct an exception    |
   | of type invalid_break_overlap() is thrown.                        |
   | ----------------------------------------------------------------- |
   | Suppose the two segments are [A1,B1] and [A2,B2]. Then, the first |
   | action for which the precondition is met is applied:              |
   |                                                                   |
   |      Precondition:              Action:                           |
   |    lists are not different    throw an exception                  |
   |    segments are not direct    throw an exception                  |
   |    segments don't overlap     return (segments are unmodified)    |
   |    A1 < A2                    [A1,B1] --> [A1,A2 - 1],[A2,B1]     |
   |    A1 > A2                    [A2,B2] --> [A2,A1 - 1],[A1,B2]     |
   |    B1 < B2                    [A2,B2] --> [A2,B1 - 1],[B1,B2]     |
   |    B1 > B2                    [A1,B1] --> [A1,B2 - 1],[B2,B1]     |
   |    segments are equal         return (segments are unmodified)    |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      28 Oct 2002         |
   ********************************************************************* */
void Qubit_list::break_overlap_with(Qubit_list &second_list,
				    iterator first_segment,
				    iterator second_segment) {
  /* be sure that the segments are direct and that
     we are not talking about the same list. */
  if ((! first_segment->is_direct())  ||
      (! second_segment->is_direct()) ||
      (&second_list == this)) throw invalid_break_overlap();
  /* take a reference to the current list. */
  Qubit_list &first_list = *this;
  /* if the two segments don't overlap, return immediately. */
  if (! first_segment->is_overlapping(*second_segment)) return;
  /* A1 < A2 : break the first list at the beginning of the second one. */
  if (first_segment->lowest() < second_segment->lowest())
    return first_list.split_at(first_segment, second_segment->lowest());
  /* A1 > A2 : break the second list at the beginning of the first one. */
  if (second_segment->lowest() < first_segment->lowest())
    return second_list.split_at(second_segment, first_segment->lowest());
  /* B1 < B2 : break the second list after the end of the first one. */
  if (first_segment->highest() < second_segment->highest())
    return second_list.split_at(second_segment, first_segment->highest() + 1);
  /* B1 > B2 : break the first list after the end of the second one. */
  if (second_segment->highest() < first_segment->highest())
    return first_list.split_at(first_segment, second_segment->highest() + 1);
  /* you can reach this point if the two segments are identical, ex-
     ception made, maybe, for their direction. Do nothing in this case. */
  return;
}

/* *********************************************************************
   | This method is the comparison operator for qubit lists, returning |
   | true if the passed list is identical to the current list, false   |
   | otherwise. The complexity is linear with the size of the lists.   |
   | ----------------------------------------------------------------- |
   | Be careful! This method tests that the two lists are identical,   |
   | i.e. made by the same sequence of the same segments. If the two   |
   | lists are minimal, this is the same thing as to say that the two  |
   | lists represent the same ordered sequence of addresses, but the   |
   | test may give a different result if minimality is not granted.    |
   | Indeed, a C++ operator== usually tests the equality of the data   |
   | structures, NOT the equivalence (i.e. that the two data structures|
   | are associated to the same mathematical object). See the method   |
   | is_equivalent() for the other test.                               |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 05 Dec 2001         |
   ********************************************************************* */
bool Qubit_list::operator==(const Qubit_list &a_list) const {
  /* take two iterators to the beginning of the lists. */
  const_iterator first_iterator  = begin();
  const_iterator second_iterator = a_list.begin();
  /* loop until one of the two iterators reaches the end of its list. */
  while (first_iterator != end() && second_iterator != a_list.end()) {
    /* return immediately if two segments are found different. */
    if (*first_iterator != *second_iterator) return false;
    /* increase both iterators. */
    ++first_iterator;
    ++second_iterator;
  }
  /* test that both iterators actually reached the end of the
     corresponding lists (otherwise one the two lists is the initial
     part of the other one, but they are not identical). */
  if (first_iterator != end() || second_iterator != a_list.end())
    return false;
  /* if one manage to get here, then the lists are equal. */
  return true;
}

/* *********************************************************************
   | This method is an interface to get_subrange() when the additional |
   | information is not needed. The arguments which must be passed by  |
   | reference are created as temporary objects which get discarded at |
   | the end. The appropriate version of get_subrange is called, de-   |
   | pending on size being 1 or not. Remember that "a_size" = 0 means  |
   | "up to the end of the list".                                      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 12 Sep 2001         |
   | Stefano Bettelli, INFN and Trento University, 13 Dec 2001         |
   ********************************************************************* */
Qubit_list Qubit_list::operator()(address an_index, size_type a_size) const {
  /* faster algorithm when size is 1 */
  if (a_size == 1) {
    /* dummy auxiliary variables */
    address dummy_address;
    const_iterator dummy_segment;
    /* call the real method with three arguments. */
    return get_subrange(an_index, dummy_address, dummy_segment);
  }
  /* standard algorithm for other sizes. */
  else {
    /* dummy auxiliary variables */
    address dummy_address_1, dummy_address_2;
    const_iterator dummy_segment_1, dummy_segment_2;
    /* call the real method with five arguments. */
    return get_subrange(an_index, a_size,
			dummy_address_1, dummy_address_2,
			dummy_segment_1, dummy_segment_2);
  }
}

/* *********************************************************************
   | This method eliminates from the current list and the passed list  |
   | all common addresses, and reorders the other so that they are in  |
   | increasing value order; in this process it can therefore modify   |
   | both lists. The return value is "true" if and only if any modifi- |
   | cation took place.                                                |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      26 Oct 2002         |
   ********************************************************************* */
//bool Qubit_list::subtract_intersection_reordering(Qubit_list &second_list) {
  /* take a reference to the current list */
//  Qubit_list &first_list = *this;
  /* reorder both lists */
//  first_list.reorder();
//  second_list.reorder();
//
//  
//}

/* *********************************************************************
   | This method is the equivalence test for qubit lists, returning    |
   | true if the passed list represents the same ordered sequence of   |
   | addresses, false otherwise. The complexity is linear with the     |
   | size of the lists.                                                |
   | ----------------------------------------------------------------- |
   | Note that this is not the same as operator==. If any list is not  |
   | minimised, a copy of it is made before testing for identity.      |
   | This method could be optimised by first checking if the two lists |
   | have the same size. Since however it is likely that the method is |
   | used for lists which are likely to be identical, I preferred to   |
   | avoid the test. Any suggestion?                                   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 05 Dec 2001         |
   ********************************************************************* */
bool Qubit_list::is_equivalent(const Qubit_list &a_list) const {
  /* check the minimality of the lists. */
  bool first_is_minimal  = is_minimal();
  bool second_is_minimal = a_list.is_minimal();
  /* if they are both minimal, the check is very easy, we can use
     the operator== which tests the identity of the data structures. */
  if (first_is_minimal && second_is_minimal) return (*this == a_list);
  /* otherwise we should make copies of the lists, minimise them and
     compare the minimised lists. This case is however unlikely since
     lists are kept minimised inside the slices. Now, initialise two
     pointers to the lists and two emtpy lists. */
  const Qubit_list *first_list  = this;
  const Qubit_list *second_list = &a_list;
  Qubit_list copy_1, copy_2;
  /* if first list is not minimal, set the pointer to a minimised copy. */
  if (! first_is_minimal)
    { copy_1 = *this; copy_1.make_minimal(); first_list = &copy_1; }
  /* do the same with the second list. */
  if (! second_is_minimal)
    { copy_2 = a_list; copy_2.make_minimal(); second_list = &copy_2; }
  /* run the identity test using the two pointers. */
  return ((*first_list) == (*second_list));
}

/* *********************************************************************
   | This method returns a Qubit_list object containing only one qubit |
   | of the current list (using an_index to select it). The segment    |
   | and address selected by the index are returned in the two addi-   |
   | tional arguments passed by reference.                             |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Feb 2001         |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */
Qubit_list Qubit_list::get_subrange(address an_index,
				    address &selected_address,
				    const_iterator &selected_segment) const {
  /* find the qubit address to which this index is translated to.
     Remember that the find_element() operation can throw an exception. */
  selected_address = find_element(an_index, selected_segment);
  /* now build a new list of Qubit_segments with a single segment;
     moreover the single segment must span the single qubit previously
     selected. We can use the custom constructor for the qubit list. 
     Then return the calculated list by value. */
  return Qubit_list(selected_address, 1);
}

/* *********************************************************************
   | This method returns a Qubit_list object which covers all the      |
   | addresses in the current list in the requested range ("a_size"    |
   | qubits using "an_index" to select the start of the range).        |
   | "a_size" = 0 means "up to the end of the list", and should be the |
   | default value. The segments and addresses selected by the indexes |
   | are returned in the four additional arguments passed by ref.      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Feb 2001         |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */
Qubit_list Qubit_list::get_subrange(address an_index, size_type a_size,
				    address &begin_address,
				    address &end_address,
				    const_iterator &begin_segment,
				    const_iterator &end_segment) const {
  /* the temporary list to be returned to the caller. Maybe putting it
     here will facilitate the named return value optimisation. */
  Qubit_list temporary_list;
  /* "a_size" = 0 conventionally means "up to the end of the list".
     Therefore now we should check for its value, and in case it is
     zero we need to calculate the real size. */
  if (a_size == 0) a_size = (size() - an_index);
  /* find the first and last qubit addresses to which this range is
     translated to. Remember that the find_element() operation can throw
     an exception. Save also the corresponding segments. Don't forget
     to subtract 1, otherwise the size would be "a_size"+1 ! */
  begin_address = find_element(an_index, begin_segment);
  end_address   = find_element(an_index + a_size - 1, end_segment);
  /* We have two different cases now; the "single segment" case requires
     a special treatement, because *begin_segment and *last_segment are
     not distinct. In this case we can simply build a list with a single
     segment covering the qubits between the two addresses (we need
     the strange constructor for Qubit_segment with the boolean argument).
     Use placement new to construct onto the temporary list. */
  if (begin_segment == end_segment)
    new (&temporary_list) Qubit_list(begin_address, end_address, true);
  /* the general case instead can copy the central part of the current
     list and add later two trimmed segments. */
  else {
    /* copy a part of the list (that between the two segments calculated
       by the translate calls, not including the segments themselves).
       It will then be adjusted to fit to the new range. Remember that
       the copied part is from the first iterator up to but not including
       the second iterator (hence advance the first segment, then go back).
       Use placement new to construct onto the temporary list. */
    ++begin_segment;
    new (&temporary_list) Qubit_list(begin_segment, end_segment);
    --begin_segment;
    /* now push to the front of the list a new segment which begins at
       begin_address and ends at begin_segment->last(), including the
       last qubit (use the strange constructor here). */
    temporary_list.push_front
      (Qubit_segment(begin_address, begin_segment->last(), true));
    /* then push to the back of the list a new segment which begins at
       end_segment->first() and ends at end_address, including the last
       qubit (use the strange constructor here). */
    temporary_list.push_back
      (Qubit_segment(end_segment->first(), end_address, true));
  }
  /* return the calculated list by value. */
  return temporary_list;
}

/* *********************************************************************
   | The split() method creates two entries from one in the list,      |
   | preserving the total size. After the method has been called, the  |
   | element pointed to by the iterator has size "new_size_first" and  |
   | it is followed by a new element with the remaining size. If the   |
   | requested new_size_first is larger than the size of first_element |
   | the operation throws exception split_error(). If new_size_first   |
   | is equal to the size of first_element, the routine does nothing.  |
   | The routine is also protected against a new size equal to zero    |
   | (it returns immediately) and against the past-the-end iterator as |
   | first_element (it throws exception split_error()).                |
   | ----------------------------------------------------------------- |
   | This method is friendly to ordered segments: it does not assume   |
   | that they are direct, but explicitely check for is_reversed().    |
   | Indeed this is done by methods operator[], last() and resize()    |
   | from the Qubit_segment class.                                     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 22 Jan 2001         |
   ********************************************************************* */
void Qubit_list::split(iterator first_element, size_type new_size_first) {
  /* new size equal to zero is a nonsense, return immediately */
  if (new_size_first == 0) return;
  /* don't be fooled by the past-the-end iterator, throw exception. */
  if (first_element == end()) throw split_error();
  /* get the old size of the referenced element. */
  size_type old_size_first = first_element->size();
  /* first check that we have enough available space, otherwise throw
     the appropriate exception. Any suggestion? */
  if (old_size_first < new_size_first) throw split_error();
  /* if size is already OK, don't mess it up (return silently). */
  if (old_size_first == new_size_first) return;
  /* since STL insertion works backward we must take an iterator to the
     first element and advance it by one. */
  iterator second_element = first_element;
  ++second_element;
  /* insert the second part of the splitted segment into the list.
     The first argument in the segment constructor is first() of the old
     segment offsetted by new_size_first (preserving the order of qubits!);
     the second argument is last() as for the old segment. The boolean in
     the ctor simply disabiguates ctor resolution. */
  insert(second_element, Qubit_segment((*first_element)[new_size_first],
				       (*first_element).last(), true));
  /* resize the first element to have the correct new size. */
  first_element->resize(new_size_first);
}

/* *********************************************************************
   | This method runs the splice() method of a generic STL list (i.e.  |
   | it moves all the elements from the second list into the current   |
   | one, in constant time; therefore none of the two lists can be     |
   | constant) and a simplification routine which might join segments  |
   | in the new list if they are contiguous. Since we assume that we   |
   | will always mantain minimal lists, the join should only happen at |
   | the border, i.e. between the last segment of the first list and   |
   | the first segment of the second one. The return value tells       |
   | whether any join took place or not. In case the two lists are not | 
   | disjoint, an exception of type invalid_splice() is immediately    |
   | thrown.                                                           |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 May 2001         |
   ********************************************************************* */
bool Qubit_list::join(Qubit_list &second_list) {
  /* don't be fooled by an empty list as argument. In this case return
     immediately with false. */
  if (second_list.is_empty()) return false;
  /* check that the two lists are not overlapping. In case they are,
     return throwing an exception of type invalid_splice(). */
  if (this->is_overlapping(second_list)) throw invalid_splice();
  /* get an iterator to the first element of the list to be "added", for
     use during the simplification stage. It is called "right side"
     because it is the right side of the junction border. */
  iterator right_side = second_list.begin();
  /* now splice the two lists, thus concatenating them. Remember however
     that the second list is empty after the splicing. */
  base_class::splice(end(), second_list);
  /* Test whether joining the segments which correspond to the two sides
     of the border is possible or not. Run the join attempt and return
     with the same exit value. Since the previously calculated iterator
     is the right side of the border we have to join backwards. */
  return try_join_and_cleanup_backward(right_side);
}

/* *********************************************************************
   | This method tries to join the element pointed to by the passed    |
   | iterator with the following one; if the join succedes, the second |
   | segment is removed and "true" is returned, otherwise "false" is   |
   | returned. The iterator must be valid; the routine is protected    |
   | against the case when either a_segment or (++a_segment) are the   |
   | past-the-end iterator end().                                      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 May 2001         |
   ********************************************************************* */
bool Qubit_list::try_join_and_cleanup_forward(iterator a_segment) {
  /* don't be fooled by the end() value for the iterator. */
  if (a_segment == end()) return false;
  /* get an iterator to the second element of the pair. This can be done
     since the iterator is not end(). Again, return immediately if the
     second iterator becomes end(). */
  iterator next_segment = a_segment;
  if ((++next_segment) == end()) return false;
  /* now both iterators are dereferenceable (ok, this is not guaranteed
     but there is nothing more we can do against dangling iterators).
     Test whether the join is possible or not. In case it is not possible,
     return immediately with "false". If the join is possible, the new
     "value" for *a_segment is already set. */
  if (! a_segment->try_join(*next_segment)) return false;
  /* if we are still here, the join was possible. The second element
     is to be removed now. The "a_segment" iterator is still valid. */
  erase(next_segment);
  /* return succesful join */
  return true;
}

/* *********************************************************************
   | This method tries to join the element pointed to by the passed    |
   | iterator with the previous one; if the join succedes, the pointed |
   | segment is removed (in this case the a_segment iterator is set to |
   | the previous element, so that it is still valid!) and "true" is   |
   | returned, otherwise "false" is returned. The iterator must be     |
   | valid; the routine is protected against the case when --a_segment |
   | is not valid. Indeed, this is only a wrapper around the basic     |
   | try_join_and_cleanup_forward.                                     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 Nov 2001         |
   ********************************************************************* */
bool Qubit_list::try_join_and_cleanup_backward(iterator a_segment) {
  /* if the passed iterator points to the first element in the list,
     there is no previous segment! Return immediately, unsuccesfully. */
  if (a_segment == begin()) return false;
  /* step back the iterator and call the forward joining. */
  bool success_status = try_join_and_cleanup_forward(--a_segment);
  /* if the join was not succesful, restore the value of a_segment. */
  if (success_status == false) ++a_segment;
  /* in any case, return the success status. */
  return success_status;
}

/* *********************************************************************
   | This method tries to join the element pointed to by the passed    |
   | iterator with both the following and the previous one; if any     |
   | join succedes, the unused segments are removed and "true" is      |
   | returned, otherwise "false" is returned. Indeed, this is only a   |
   | wrapper around try_join_and_cleanup_forward which is called twice.|
   | The iterator must be valid; the routine is protected against the  |
   | case when either central_element or (++central_element) are the   |
   | past-the-end iterator, and when (--central_element) is not valid. |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 30 Nov 2001         |
   ********************************************************************* */
bool Qubit_list::try_join_and_cleanup_both_sides(iterator a_segment) {
  /* this boolean variable will remember if any join took place. */
  bool any_join = false;
  /* try a join between the segment and its follower. The called routine
     is already protected against the possibility that "central_element" is
     the past-the-end iterator, so don't check it twice. */
  any_join = any_join || try_join_and_cleanup_forward(a_segment);
  /* do the same thing with the previous segment. */
  any_join = any_join || try_join_and_cleanup_backward(a_segment);
  /* return the boolean variable. */
  return any_join;
}

/* *********************************************************************
   | This method takes care to translate an index inside a list of     |
   | addresses into an address, i.e. it finds the address of the       |
   | index-th location in the list. If the index is out of range,      |
   | exception out_of_range() is thrown. The range of valid indexes is |
   | 0, 1, ..., size()-1. The second argument (an iterator inside the  |
   | address list, passed by reference) is pointed to the segment in   |
   | which the address was found.                                      |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Feb 2001         |
   ********************************************************************* */
Qubit_list::address Qubit_list::find_element
(address an_index, const_iterator &a_segment) const {
  /* start the search from the very beginning of the qubit list. */
  a_segment = begin();
  /* call a more specific routine which does all the hard job.
     a_segment will be modified during its execution. An exception may
     be thrown. Subtract the return value from an_index, so that it will
     be an index in the modified a_segment. */
  an_index -= find_segment(an_index, a_segment);
  /* ask the segment to provide the translation and return it. */
  return (*a_segment)[an_index];
}

/* *********************************************************************
   | This method shifts a part of a list forward. The first "head" ad- |
   | dresses are left unaltered, while the others are shifted forward  |
   | by "jump" units (have a look at the Qop requirements for details  |
   | about why this method is useful). Remember that if the jump can't |
   | be executed because the addresses in the segment would overflow,  |
   | the segment class will throw the appropriate exception.           |
   | ----------------------------------------------------------------- |
   | (07 Jun 2001) S.Bettelli. It is possible for at most one segment  |
   | to be broken by the split routine. This happens if the segment is |
   | not completely inside the first "head" locations nor completely   |
   | outside. This breaking MUST NOT assume that the segment is direct |
   | (i.e. it must preserve the order of segment locations).           |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 May 2001         |
   ********************************************************************* */
void Qubit_list::run_split(size_type head, size_type jump) {
  /* loop through the list analyzing each segment separately. */
  for (iterator a_segment = begin(); a_segment != end(); ++a_segment)
    /* if the current segment is completely within the first "head"
       addresses, we don't care. Therefore test for at least one
       address (the highest) being greater or equal to head. */
    if (a_segment->highest() >= head) {
      /* if the segment is not completely within the first "head"
	 addresses nor completely outside, it is to be shifted 
	 partially. In order to achieve this goal, we will split
	 it into two parts, preserving the order. The second part
	 will be processed during the next loop iteration. The 
	 return value from split_at() is saved in broken_segment. */
      if (a_segment->lowest() < head) split_at(a_segment, head);
      /* if the current segment is (now?) completely beyond the first
	 "head" addresses, increase the addresses for each segment
	 by "jump" units and move to the next segment. */
      if (a_segment->lowest() >= head) a_segment->shift_forward(jump);
    }
}

/* *********************************************************************
   | This method inverts some formal locations inside a list without   |
   | changing their "order". The first "head" addresses as well as     |
   | those from the "head+size"-th one onward are left unaltered,      |
   | while the others have their value interchanged. Have a look at    |
   | the requirements for Qop for more details.                        |
   | ----------------------------------------------------------------- |
   | The first part of the routine may break up some segments in order |
   | to have each segment contained in exactly one of the three parts. |
   | After this, each segment in the central part, defined by the      |
   | range [head, head+size[, has its address recalculated around the  |
   | barycenter of this range. The order of transformed locations is   |
   | left unaltered. The list is then reduced to minimality again.     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 29 May 2001         |
   ********************************************************************* */
void Qubit_list::run_invert(size_type head, size_type size) {
  /* Calculate the first address in the second part of the list
     which must be left unaltered. Throw an exception in case the
     "size" variable causes an address overflow. */
  size_type tail = head + size;
  if (tail < head) throw invert_failed();
  /* initialise two iterators to the past-the-end value for the list.
     These iterators will be used for remembering segments in the
     "central" part of the list which could be joined with other ones
     after they have been modified. Actually, there are only two
     borders ([head-1,head] and [tail-1,tail]) where a join can
     happen; moreover, since the list is assumed to be overlap free,
     no two segments can have lowest() = head (or highest() = tail-1).
     Hence we can get by with only two iterators: remember though 
     that they could point to the same segment! */
  iterator lower_border = end();
  iterator upper_border = end();
  /* Loop over all segments in the list. */
  for (iterator a_segment = begin(); a_segment != end(); ++a_segment) {
    /* if the current segment starts within the first "head" addresses
       but is not completely contained in that range, we will split it
       into two parts, preserving the order; the second part will be
       processed during the next loop iteration. */
    if ((a_segment->lowest() < head) && (a_segment->highest() >= head))
      split_at(a_segment, head);
    /* do the same splitting in case we are crossing the border
       between tail-1 and tail. */
    if ((a_segment->lowest() < tail) && (a_segment->highest() >= tail))
      split_at(a_segment, tail);
    /* now we are sure that the current segment is completely contained
       either in the two list parts to be left untouched (i.e. the range
       [0, head[ and the range [tail, +infty[) or in the central part
       which must be "inverted". */
    if ((a_segment->lowest() >= head) && (a_segment->highest() < tail)) {
      /* ok, this is a segment which must be "inverted". Calculate the
	 new lowest address for this segment. Since we are performing
	 a mirror simmetry, it is the transformed address corresponding
	 to the "old" highest address. Remember -1 */
      address new_lowest = head + (tail - a_segment->highest()) - 1;
      /* The size of the segment is not going to change, therefore we
	 can perform the job with a simple conditional shift. But we
	 have to reverse the order inside the segment afterward, otherwise
	 the relative order of transformed addresses would be changed. */
      a_segment->shift_lowest(new_lowest);
      a_segment->reverse();
      /* if, after this transformation, the segment has lowest() = head,
	 then its iterator must be saved in the appropriate variable. */
      if (a_segment->lowest() == head) lower_border = a_segment;
      /* do the analogous thing if highest() = tail - 1 */
      if (a_segment->highest() == (tail - 1)) upper_border = a_segment;
    }
  }
  /* we are going to minimise the list again now. If both iterators
     point to the same segment, suppress one of the two, because it
     could become invalid in the meanwhile. */
  if (lower_border == upper_border) upper_border = end();
  /* try to run a join + cleanup with the previous and next segments
     for both the lower_border and the upper_border. The called
     routine is protected against everything, including the iterators
     being the past-the-end iterators, therefore don't worry. */
  try_join_and_cleanup_both_sides(lower_border);
  try_join_and_cleanup_both_sides(upper_border);
}

/* *********************************************************************
   | This method performs a remapping of the current list using a per- |
   | mutation list. The definition is as follows: If the i-th element  |
   | of the old current list is j, then the i-th element of the new    |
   | current list is the j-th element of the passed "permutation_list",|
   | which acts like a permutation. This is a sort of generalisation   |
   | of run_split and run_invert (have a look at the Qop requirements  |
   | for details about why this method is useful).                     |
   | Of course, if the maximum index contained in the current list is  |
   | m, then the permutation_list must have at least (m+1) elements,   |
   | otherwise an exception of type invalid_translation is thrown.     |
   | ----------------------------------------------------------------- |
   | This method is nothing more than an interface to translate(),     |
   | where the interested reader can find more details.                |
   |                                                                   |
   |  Stefano Bettelli, IRSAMC, UPS, Toulouse,       13 May 2002       |
   ********************************************************************* */
void Qubit_list::run_remap(const Qubit_list &permutation_list) {
  /* Use the translate() method to do all the dirty job. Unluckily,
     this method is not as efficient as run_split and run_invert,
     because an intermediate list must be created (this is the
     way translate() works ...). */
  Qubit_list new_current_list = translate(permutation_list);
  /* swap the intermediate list into the current list. */
  swap(new_current_list);
}

/* *********************************************************************
   | This method takes care to find the segment in the current list    |
   | which contains the index-th list location, assuming that the      |
   | first location in the supplied "a_segment" has index 0. If the    |
   | index is out of range, exception out_of_range() is thrown.        |
   | ----------------------------------------------------------------- |
   | an_index: is the index of the location to be found, with respect  |
   |           to the first() location of the second argument.         |
   | a_segment: this is the segment iterator from which the search is  |
   |            started. Its value is modified internally so that at   |
   |            the end it will point to the segment containing the    |
   |            searched location.                                     |
   | The return value is the number of visited locations, i.e. the     |
   | accumulated sizes of all segments in [a_segment, a_segment'[,     |
   | where a_segment' is the value of a_segment after execution of     |
   | this routine.                                                     |
   | ----------------------------------------------------------------- |
   | Finding the address means iterating through the segments in the   |
   | list (accumulating the number of qubits inside them) until we     |
   | reach the segment covering the index. Of course we could know in  |
   | advance whether the index is out of range or not, but since this  |
   | is a run-time error it would be a nonsense to optimise for it.    |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Jul 2001         |
   ********************************************************************* */
Qubit_list::size_type
Qubit_list::find_segment(address an_index, const_iterator &a_segment) const {
  /* a temporary variable for holding the initial value of an_index. */
  size_type initial_index = an_index;
  /* analyze each segment in sequence until you find the searched one
     or you bump into the past-the-end segment value. */
  while (a_segment != end()) {
    /* if index is small "enough" we found the segment, stop iterating. */
    if (an_index < a_segment->size()) break;
    /* otherwise decrease index by the number of already skipped locations. */
    else an_index -= a_segment->size();
    /* in any case, increase the iterator. */
    ++a_segment;
  }
  /* check the way we exited the loop. If the list scan ended "normally", 
     i.e. reaching the very end of the list, then the index could not be
     found in any segment, i.e. it was out of range --> throw exception. */
  if (a_segment == end()) throw out_of_range();
  /* return the accumulated sizes for skipped segments. */
  return (initial_index - an_index);
}

/* *********************************************************************
   | This method accepts a list object (the "address list") and builds |
   | a copy of the current list (the "index list") replacing each      |
   | index i with the i-th location in the address list. This method   |
   | will be used at "execution time" by operators which need to tran- |
   | slate their index list into a real location list by fitting it    |
   | onto a quantum register.                                          |
   | ----------------------------------------------------------------- |
   | This routine works by matching each possible pair of one segment  |
   | from the current list and one segment from the address_list. It   |
   | is fast when the number of segments is small, but it incurs a     |
   | quadratic slowdown asymptotically (in this case it is better to   |
   | use "translate_map"). The inline "translate" method tries to se-  |
   | lect that routine among these two which is best fit for the task. |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      22 Jul 2002         |
   ********************************************************************* */
Qubit_list Qubit_list::translate_nomap(const Qubit_list &address_list) const {
  /* This is the list to be returned at the end of the routine,
     containing the translated addresses. Put it here to facilitate
     the named return value optimisation. */
  Qubit_list translated_list;
  /* loop over all segments in the index list (i.e. *this) */
  for (const_iterator index_segment = begin();
       index_segment != end(); ++index_segment) {
    /* translate the current index segment (this is delegated to another
       method, resetting the search every time, since we are not guaranteed
       that the index segments are selected in increasing value order). The
       translation list is returned as a dynamically allocated object. */
    Qubit_list *temporary_list =
      translate_segment(address_list, index_segment, true);
    /* splice the content of this list at the end of the overall list. */
    translated_list.splice(translated_list.end(), *temporary_list);
    /* release the dinamically allocated memory. */
    delete temporary_list;
  }
  /* also run a minimisation routine to reduce the list to minimality
     (it is not guaranteed that it is minimal now). */
  translated_list.make_minimal();
  /* return the translated list. */
  return translated_list;
}

/* *********************************************************************
   | This private method is common to all the translate_* methods. It  |
   | returns a dynamically allocated Qubit_list corresponding to the   |
   | translation of the index_segment of the current list into the     |
   | address_list. The "reset" boolean variable controls the behaviour |
   | of two static variables which remember data about the last search;|
   | when it is true the search is restarted from scratch (which is    |
   | the safe choice when you don't know what you are doing).          |
   | ----------------------------------------------------------------- |
   | In practice the search shouldn't be reset (exception made for the |
   | first segment) only when the index_segments are presented in in-  |
   | creasing value order (like in translate_map).                     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      28 Jan 2003         |
   ********************************************************************* */
Qubit_list *Qubit_list::translate_segment(const Qubit_list &address_list,
					  const const_iterator &index_segment,
					  bool reset) const {
  /* "address_segment" is the segment in the address list which we last
     analised, while "visited_locations" is an accumulator which keeps
     track of all the sizes of already analyzed segments in the address
     list. It is sometimes useful to restart from these old values. */
  static const_iterator address_segment   = address_list.begin();
  static size_type      visited_locations = 0;
  /* If "reset" is true, we have to erase the memory of the past searches,
     i.e. set address_segment to the beginning of the address_list and
     therefore visited_locations to zero. */
  if (reset) { address_segment = address_list.begin(); visited_locations = 0; }
  /* read and save the lowest and highest addresses in the
     segment pointed to by "index_segment" once for all. */
  const address lowest_index  = index_segment->lowest();
  const address highest_index = index_segment->highest();
  /* scan the address list in order to find the segment which hosts the
     location corresponding the lowest_index. Start from the position
     selected by address_segment, syncronised with visited_locations. */
  try { visited_locations += address_list.find_segment
	  (lowest_index - visited_locations, address_segment); }
  /* translate the exception, if it occurs. */
  catch (out_of_range) { throw invalid_translation(); }
  /* calculate the address corresponding to the index and save the
     current value for the address_segment iterator. */
  const const_iterator low_address_segment = address_segment;
  const size_type low_address =
    (*low_address_segment)[lowest_index - visited_locations];
  /* run the same search for the highest address in the index_segment. */
  try { visited_locations += address_list.find_segment
	  (highest_index - visited_locations, address_segment); }
  /* translate the exception, if it occurs. */
  catch (out_of_range) { throw invalid_translation(); }
  /* calculate the address corresponding to the index. */
  const const_iterator high_address_segment = address_segment;
  const size_type high_address =
    (*high_address_segment)[highest_index - visited_locations];
  /* translate the current index segment. This is done by the extract()
     method, which needs low_address and high_address (elements of the
     address list), the iterators to the segments into the address list
     where they can be found, and the directness of index_segment. The
     translation list is returned as a dynamically allocated object. */
  return extract(low_address, high_address,
		 low_address_segment, high_address_segment,
		 index_segment->is_reversed());
}

/* *********************************************************************
   | This method returns a section of the current list, namely that    |
   | bordered by the two elements low_address and high_address (remem- |
   | ber, these are elements of the current list, not indexes inside   |
   | it!). The iterators low_address_segment and high_address_segment  |
   | point to the segments which contain these two border elements.    |
   | The result is returned as a dinamically allocated qubit list.     |
   | If "reverse" is true, the returned list is also reversed.         |
   | ----------------------------------------------------------------- |
   | First of all, try an optimisation for the case when the low and   |
   | the high segments are the same. This always ends up in the selec- |
   | ted sublist being a single segment (this can be performed with a  |
   | single ctor). For the general case, the strategy is different:    |
   | prepare copies of the concerned segments in the current list (al- |
   | ready having the correct segmentation), then adjust the borders.  |
   | ----------------------------------------------------------------- |
   | (28 Jan 2003) S.Bettelli, reworked as a more general routine.     |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 31 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      28 Jan 2003         |
   ********************************************************************* */
Qubit_list *Qubit_list::extract(address low_address, address high_address,
				const_iterator low_address_segment,
				const_iterator high_address_segment,
				bool reverse) const {
  /* try the optimisation for the single segment case. */
  if (low_address_segment == high_address_segment) {
    /* take different actions depending on "direct". */
    if (reverse) return new Qubit_list(high_address, low_address, true);
    else         return new Qubit_list(low_address, high_address, true);
  }
  /* The general case begins here: remember that we are now sure that the
     two segments are not coincident. Calculate the last element of the
     low_address_segment and the first element of the high_address_segment. */
  const address low_address_segment_last   = low_address_segment->last();
  const address high_address_segment_first = high_address_segment->first();
  /* Range-copy most (?) of the segments to be extracted into an auxiliary
     list. This, actually, must not include the segments pointed to by
     low_address_segment and high_address_segment (it means that the
     sublist can be empty, but this is not a problem). The first iterator
     is modified for efficiency, remember not to use it anymore! */
  Qubit_list *sublist =
    new Qubit_list(++low_address_segment, high_address_segment);
  /* Add the elements of the low_address_segment, from low_address
     up to its last element, in front of the sublist. */
  sublist->insert(sublist->begin(),
		  Qubit_segment(low_address, low_address_segment_last, true));
  /* Add the elements of the high_address_segment, from its first
     element up to the high_address, at the end of the sublist. */
  sublist->insert(sublist->end(),
		  Qubit_segment(high_address_segment_first,high_address,true));
  /* reverse the extracted sublist if required. */
  if (reverse) sublist->reverse();
  /* return the dinamically allocated list */
  return sublist;
}

/* *********************************************************************
   | This static method takes a collection of address lists and mixes  |
   | them, appending the result to the returned object. This method is |
   | used by the quantum interface at execution time.                  |
   | ----------------------------------------------------------------- |
   | The supplied lists will be splitted in order to have the same     |
   | number of segments (with corresponding segments having the same   |
   | size); subsequently, their segments will be moved to the list to  |
   | be returned so that corresponding segments will be consecutive.   |
   | In this case the complexity of this operation is proportional     |
   | to the product of the number of lists and the total number of     |
   | segments in these lists. If the lists were not all of the same    |
   | size, a check at the end of the routine will issue an exception   |
   | of type invalid_join().                                           |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 01 Aug 2001         |
   ********************************************************************* */
Qubit_list Qubit_list::mix_lists(Qubit_list_collection &a_collection) {
  /* prepare the list to be returned here, in order to facilitate
     the name return value optimisation. */
  Qubit_list mixed_list;
  /* save the number of lists in the passed collection. */
  typedef Qubit_list_collection::size_type list_size_type;
  list_size_type number_of_lists = a_collection.size();
  /* now we need a recipe for mixing up the lists. Our strategy will be
     the following: 1) calculate the minimum size among first segments
     in each list 2) split each of the first segments to fit this size
     3) splice all first segments onto the list to be returned 4) repeat
     until passed lists are emtpy. */
  while (! a_collection[0].empty()) {
    /* initialise the minimum size for first segments */ 
    size_type min_size = a_collection[0].front().size();
    /* loop over all other lists (don't lookup the first again) */
    for (list_size_type index = 1; index < number_of_lists; ++index) {
      /* stop immediately if min_size is already minimal (optimisation) */
      if (min_size == 1) break;
      /* get the size of the first segment of the selected list */
      size_type current_size = a_collection[index].front().size();
      /* update min_size if necessary */
      if (current_size < min_size) min_size = current_size;
    }
    /* loop again (this time including the first list). */
    for (list_size_type index = 0; index < number_of_lists; ++index) {
      /* split the first segment of the selected list to fit to the
	 requested size (if size is already ok this is a no-op) */
      a_collection[index].split(a_collection[index].begin(), min_size);
      /* append the first segment, which now has size min_size,
	 to the list to be returned. */
      mixed_list.splice(mixed_list.end(), a_collection[index],
			a_collection[index].begin());
    }
  }
  /* run a sanity check: all lists should be empty now */
  for (list_size_type index = 0; index < number_of_lists; ++index)
    if (! a_collection[index].empty()) throw invalid_join();
  /* return the mixed list. */
  return mixed_list;
}

/* *********************************************************************
   | This output function will display the content of the list, for    |
   | debugging purposes. The header line summarises a list identifier  |
   | (the address, sic!) and the number of segments and qubits inside. |
   | Then each segment is output on a separate line through its own    |
   | output function.                                                  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 17 Jan 2001         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const Qubit_list &a_list) {
  /* first, print a list identifier (the pointer, sich..) */
  os << "\t\tList " << static_cast<const void *>(&a_list)
    /* now print the number of segments in the list. */
     << " contains " << a_list.segments() << " segment(s)"
    /* now print the list "size" (i.e. the number of qubits in the list */
     << " and " << a_list.size() << " qubit(s)";
  /* This variable is a shortcut for indexing segments. */
  Qubit_list::size_type segment_counter = 0;
  /* Now iterate over all segments and call the appropriate output function */
  for (Qubit_list::const_iterator segment = a_list.begin();
       segment != a_list.end(); ++segment, ++segment_counter)
    os << "\n\t\t  (" << segment_counter << ")\t" << *segment;
  /* return the stream */
  return os;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
