/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <vector>                                 // STL vectors
#include "qop_builders.h"                         // function prototypes
#include "qop_primitives.h"                       // primitive slices

/* a macro which save some typing in the following */
#define push(the_slice) push_back(new the_slice)

/* *********************************************************************
   | This routine will build a circuit specification for a negation    |
   | gate (NOT, i.e. the Pauli X matrix) on the qubits specified by    |
   | "the_list". Each NOT is converted into two Hadamard gates and a   |
   | phase shift with power = 1 (i.e. a Pauli Z gate).                 |
   | ----------------------------------------------------------------- |
   | The identity which is relevant here is HRH = X where H is the     |
   | Hadamard matrix, X is the first Pauli matrix (quantum "not") and  |
   | R is a phase shift with angle equal to pi (which by the way is    |
   | just Z, the third Pauli matrix). Of course HH = 1.                |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      12 Jun 2002         |
   ********************************************************************* */
QNot_slices::QNot_slices(const Qubit_list &the_list) {
  /* first layer with Hadamard gates. */
  push(Qop_slice_Hadamard(the_list));
  /* second layer with PI shifts (phase = -1) */
  push(Qop_slice_Phase(the_list, 1));
  /* third layer with Hadamard gates. */
  push(Qop_slice_Hadamard(the_list));
}

/* *********************************************************************
   | This routine will build a circuit specification for a conditional |
   | negation (CNOT) on the qubits specified by "the_controls" and     |
   | "the_targets". Each CNOT is converted into two Hadamard gates and |
   | a conditional phase shift with power = 1.                         |
   | ----------------------------------------------------------------- |
   | The identity which is relevant here is HRH = X where H is the     |
   | Hadamard matrix, X is the first Pauli matrix (quantum "not") and  |
   | R is a phase shift with angle equal to pi (which by the way is    |
   | just Z, the third Pauli matrix). Of course HH = 1.                |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */
QCnot_slices::QCnot_slices(const Qubit_list &the_controls,
			   const Qubit_list &the_targets) {
  /* first layer with Hadamard gates on targets. */
  push(Qop_slice_Hadamard(the_targets));
  /* second layer with controlled PI shifts (phase = -1) */
  push(Qop_slice_CondPhase(the_controls, the_targets, 1));
  /* third layer with Hadamard gates on targets. */
  push(Qop_slice_Hadamard(the_targets));
}

/* *********************************************************************
   | This routine will build a circuit specification for a quantum     |
   | Fourier transform on the qubits specified by "the_list".          |
   | Remember that the most significant bit (MSB) is the first in the  |
   | passed list, consistently with the list class and the segment     |
   | class. The order is unaltered after application of the circuit.   |
   | ----------------------------------------------------------------- |
   | The circuit which is built here is the classical quadratic cir-   |
   | cuit which is the quantum version of the fast Fourier transform,  |
   | as in D.Coppersmith, "An Approximate Fourier Transform Useful in  |
   | Quantum Factoring", Technical report IBM, Research report 19642,  |
   | IBM, 07/12/1994. The order of qubits is swapped at the end, so    |
   | that we comply with the requirement that the MSB is always the    |
   | first in the qubit list.                                          |
   | (30 Nov 2001) S.Bettelli, introduced a real software swap.        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */
QFourier_slices::QFourier_slices(const Qubit_list &the_list) {
  /* get the number of qubits in the list. */
  size_type list_size = the_list.size();
  /* prepare an array for holding each qubit separately as a list. */
  std::vector<const Qubit_list *> qubits(list_size);
  /* dinamically allocate a list object for each index, which must
     contain the qubit in the_list referenced by that index. */
  for (size_type index = 0; index < list_size; ++index)
    qubits[index] = new Qubit_list(the_list(index, 1));
  /* iterate over all indexes and use them as targets. */
  for (size_type target = 0; target < list_size; ++target) {
    /* first, run a Hadamard transformation on each line */
    push(Qop_slice_Hadamard(*qubits[target]));
    /* then iterate on all other indexes and use them as controls. */
    for (size_type control = (target + 1); control < list_size; ++control)
      /* for each control and target, push a conditional phase shift
	 with an angle of 2PI/2^k, k being the distance between the
	 control and the target plus 1. */
      push(Qop_slice_CondPhase(*qubits[control], *qubits[target],
			       control - target + 1));
  }
  /* deallocate all the dinamically allocated auxiliary lists. */
  for (size_type index = 0; index < list_size; ++index)
    delete qubits[index];
  /* add the swapping all the qubits. The swap slice needs two lists,
     which are the first half of "the_list" and the reversed second
     part of "the_list". Check that the list_size is at least 2 before
     trying to push this swap. */
  if (list_size > 1)
    push(Qop_slice_Swap(the_list(0, list_size/2),
			the_list(list_size - list_size/2, 0).reverse()));
}

/* *********************************************************************
   | This routine will build a circuit specification for a conditional |
   | Hadamard gate on the qubits specified by "the_controls" and       |
   | "the_targets". Each controlled Hadamard is converted into three   |
   | controlled rotations, one unconditioned rotation and two Hadamard |
   | gates.                                                            |
   | ----------------------------------------------------------------- |
   | This circuit has been calculated using a decomposition into Euler |
   | angles around the z and x (not y!) axis: H = i R_z(a)R_x(a)R_z(a) |
   | where a=PI/2. The x-rotation in the middle is then converted with |
   | the identity R_x(a)=HR_z(a)H. R_z(PI/2) is proportional to R_2.   |
   | It can be shown that R_2 H R_2 H R_2 = e^(iPI/4)H. By controlling |
   | the three phase shifts we get the identity when the control line  |
   | is found in |0> and almost H when it is found in |1> : the uncon- |
   | ditional rotation on the control line corrects for the additional |
   | global phase on the target line.                                  |
   |     push(Qop_slice_CondPhase(the_controls, the_targets, 2));      |
   |     push(Qop_slice_Hadamard(the_targets));                        |
   |     push(Qop_slice_CondPhase(the_controls, the_targets, 2));      |
   |     push(Qop_slice_Hadamard(the_targets));                        |
   |     push(Qop_slice_CondPhase(the_controls, the_targets, 2));      |
   |     push(Qop_slice_Phase(the_controls, 3, -1.0));                 |
   | ----------------------------------------------------------------- |
   | (01 Nov 2002) S.Bettelli. I use now another decomposition for a   |
   | controlled Hadamard, inspired by quant-ph/0207157 (G.Song and A.  |
   | Klappenecker, "Optimal realizations of controlled unitary gates") |
   | The relevant identity is R_2 H R_3 H Z H R_3* H R_2* = H, and the |
   | controlled Hadamard can be obtained by controlling the Z in the   |
   | middle of the expression. Note that this decomposition uses 8 one |
   | qubit gates but only 1 two qubit gate (the previous construction  |
   | involved 3 one qubit gates and 3 two qubit gates), which could be |
   | an improvement, since two qubit gates are usually regarded as     |
   | more difficult to implement in a real quantum computer.           |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 26 Sep 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      01 Nov 2002         |
   ********************************************************************* */
QCondHadamard_slices::QCondHadamard_slices(const Qubit_list &the_controls,
					   const Qubit_list &the_targets) {
  /* a sequence with R_2 H R_3 H on the target qubits. */
  push(Qop_slice_Phase(the_targets, 2));
  push(Qop_slice_Hadamard(the_targets));
  push(Qop_slice_Phase(the_targets, 3));
  push(Qop_slice_Hadamard(the_targets));
  /* the single controlled operation (a controlled Z) */
  push(Qop_slice_CondPhase(the_controls, the_targets, 1));
  /* the adjoint of R_2 H R_3 H on the target qubits. */
  push(Qop_slice_Hadamard(the_targets));
  push(Qop_slice_Phase(the_targets, 3, -1.0));
  push(Qop_slice_Hadamard(the_targets));
  push(Qop_slice_Phase(the_targets, 2, -1.0));
}

/* *********************************************************************
   | This routine will build a circuit specification for a doubly      |
   | conditional Phase gate on the qubits specified by the two control |
   | lists and "the_targets". Each doubly controlled Phase is conver-  |
   | ted into five controlled phase shifts and four Hadamard gates.    |
   | ----------------------------------------------------------------- |
   | [This is an old note, the circuit is in my PhD thesis]            |
   | I copied this circuit from the famous book by Nielsen and Chuang, |
   | "Quantum Computation and Quantum Information", Cambridge Univer-  |
   | sity Press, 2000, at pag. 182, and dropped some terms acting on   |
   | the two controls only. I don't know who first elaborated this     |
   | circuit (if anyone knows it, please drop me a line!).             |
   | Basically we have four CNOTs with alternated controls, separated  |
   | by four higher order phase shifts. The final controlled gate      |
   | between the two controls corrects a phase.                        |
   | ----------------------------------------------------------------- |
   | (30 Apr 2002) S.Bettelli; this circuit construction is valid even |
   | when the phase is exp(2 PI i x / 2^k) with x generic (i.e. not 1) |
   | but we need to pass explicitely the value of x to the conditional |
   | phase gate constructors.                                          |
   | ----------------------------------------------------------------- |
   | (11 Jul 2002) S.Bettelli; I've switched to a much better circuit, |
   | suggested by Klaus Frahm (IRSAMC, UPS, Toulouse) during a talk    |
   | today at the Toulouse QUANTWARE workshop. It uses only three      |
   | conditional phase gates and two CNOTs (hence 9 gates with our     |
   | current approach); moreover, I only need angles which are 1/2 of  |
   | of the passed one, not also 1/4 as before.                        |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 26 Sep 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      30 Apr 2002         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      11 Jul 2002         |
   ********************************************************************* */
QCondCondPhase_slices::
QCondCondPhase_slices(const Qubit_list &the_controls_1,
		      const Qubit_list &the_controls_2,
		      const Qubit_list &the_targets,
		      ::quantum_phase_power_type    the_power,
		      ::quantum_phase_floating_type the_fraction) {
  /* push two controlled phase gate with power = k+1; first act on
     controls 1 & 2, then act on controls 2 and targets. */
  push(Qop_slice_CondPhase(the_controls_1, the_controls_2,
			   the_power+1, the_fraction));
  push(Qop_slice_CondPhase(the_targets, the_controls_2,
			   the_power+1, the_fraction));
  /* push a CNOT circuit between the_controls_1 and the_targets. */
  splice(~QCnot_slices(the_controls_1, the_targets));
  /* push a controlled phase gate with power = k+1 on controls 2 &
     targets (it is the adjoint of the previous controlled phase gate) */
  push(Qop_slice_CondPhase(the_targets, the_controls_2,
			   the_power+1, -the_fraction));
  /* push a CNOT circuit between the_controls_1 and the_targets. */
  splice(~QCnot_slices(the_controls_1, the_targets));
}

/* *********************************************************************
   | This routine will build a circuit specification for the Toffoli   |
   | gate, which can be used to implement multicontrolled gates. This  |
   | gate acts just like the indentity on three qubits, exception made |
   | when both the control qubits are |1>, in which case acts like X.  |
   | The names of the arguments are self explanatory.                  |
   | ----------------------------------------------------------------- |
   | Building a Toffoli gate is simple once we have the doubly con-    |
   | trolled phase shifts. Enclosing a doubly controlled Z=R_1 between |
   | two Hadamard gates is enough, since HH = 1 and HZH = X.           |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 11 Sep 2001         |
   ********************************************************************* */
Toffoli_slices::Toffoli_slices(const Qubit_list &the_controls_1,
			       const Qubit_list &the_controls_2,
			       const Qubit_list &the_targets) {
  /* push the initial Hadamard gates. */
  push(Qop_slice_Hadamard(the_targets));
  /* build a doubly controlled phase shift with k=1 and x = 1.0 */
  splice(~QCondCondPhase_slices(the_controls_1, the_controls_2,
				the_targets, 1, 1.0));
  /* push the final Hadamard gates. */
  push(Qop_slice_Hadamard(the_targets));
}

/* *********************************************************************
   | This routine will build a circuit specification for the condiito- |
   | ned swap gate. This circuit will exchange the physical state of   |
   | the two "swaplist"s if the control register is high. Practically, |
   | this is like controlling three CNOT gates (a controlled CNOT gate |
   | is like a Toffoli gate).                                          |
   | ----------------------------------------------------------------- |
   | The "uncontrolled" swap gate can be implemented in software,      |
   | since it is a completely classical operation. A controlled swap   |
   | however is not classical and must be executed on the q. device.   |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 03 Dec 2001         |
   ********************************************************************* */
QCondSwap_slices::QCondSwap_slices(const Qubit_list &the_controls,
				   const Qubit_list &the_swaplist_1,
				   const Qubit_list &the_swaplist_2) {
  /* push three Toffoli gates, the "target" register is the second,
     then the first, then the second swap list again. */
  splice(~Toffoli_slices(the_controls, the_swaplist_1, the_swaplist_2));
  splice(~Toffoli_slices(the_controls, the_swaplist_2, the_swaplist_1));
  splice(~Toffoli_slices(the_controls, the_swaplist_1, the_swaplist_2));
}
		
//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
