/* *********************************************************************
   | The Q language - A C++ extension for programming quantum machines |
   | Copyright (C) 2000 2001 2002 2003 Stefano Bettelli                |
   | <bettelli@irsamc.ups-tlse.fr>                                     |
   | See the COPYING and LICENSE files for license terms.              |
   ********************************************************************* */
#include <iomanip>                              // C++ I/O manipulators
#include "qop_slice.h"                          // class declaration

/* *********************************************************************
   | This is the default (protected) constructor for a time slice.     |
   | It sets the base gate pointer and initialises the internal lists  |
   | using those supplied by the user. A check is subsequently run in  |
   | order to be sure that different lists are not overlapping.        |
   | Moreover, all the lists must have the same size. Any failure to   |
   | comply with these requirements will result in the throwing of an  |
   | exception of type invalid_list(). Each list is then simplified to |
   | minimal terms by a join routine.                                  |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 12 Jun 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      10 Nov 2002         |
   ********************************************************************* */
Qop_slice::Qop_slice(const Qop_slice_basegate *the_basegate,
		     const collection_type &user_lists) :
  basegate(the_basegate),               // set the associated base gate
  index_lists(user_lists.size()) {      // allocate enough dummy lists
  /* Copy the user lists in the internal collection. Use placement new
     to minimise the number of copies (empty lists are already present
     in the internal collection). */
  index_type index = 0;
  for (collection_type::const_iterator a_list = user_lists.begin();
       a_list != user_lists.end(); ++a_list, ++index)
    new (&get_list(index)) list_type(*a_list);
  /* Get size of the first list. If everything is correct, all the
     other lists must have the same size. */
  list_type::size_type common_size = get_list(0).size();
  /* Run the consistency tests and the minimisation routine. */
  for (index_type index = 0; index < get_list_number(); ++index) {
    /* check that the size is correct */
    if (get_list(index).size() != common_size) throw invalid_list();
    /* reduce the list to minimality. */
    get_list(index).make_minimal();
  }
  /* now check that each pair of lists has empty intersection. Throw an
     exception for invalid list if any non-empty intersection is found. */
  for (index_type index_1 = 0; index_1 < get_list_number(); ++index_1)
    for (index_type index_2=(1+index_1); index_2< get_list_number(); ++index_2)
      if (get_list(index_1).is_overlapping(get_list(index_2)))
	throw invalid_list();
}

/* *********************************************************************
   | This constructor is a specialisation for the case where the user  |
   | supplies only one index list. Most tests can be skipped in this   |
   | case. See the more general constructor which accepts a const re-  |
   | ference to a collection_type object for further details.          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 25 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      10 Nov 2002         |
   ********************************************************************* */
Qop_slice::Qop_slice(const Qop_slice_basegate *the_basegate,
		     const list_type &the_user_list) :
  basegate(the_basegate),         // set the associated base gate
  index_lists(1) {                // prepare a dummy empty list
  /* copy the user list with placement new. */
  new (&get_list(0)) list_type(the_user_list);
  /* reduce the list to minimality. */
  get_list(0).make_minimal();
}

/* *********************************************************************
   | This constructor is a specialisation for the case where the user  |
   | supplies only two index lists. Most tests can be skipped in this  |
   | case. See the more general constructor which accepts a const re-  |
   | ference to a collection_type object for further details.          |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 25 Jul 2001         |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      10 Nov 2002         |
   ********************************************************************* */
Qop_slice::Qop_slice(const Qop_slice_basegate *the_basegate,
		     const list_type &user_list_1,
		     const list_type &user_list_2) :
  basegate(the_basegate),         // set the associated base gate
  index_lists(2) {                // create two dummy empty lists
  /* copy the two lists into the list container */
  new (&get_list(0)) list_type(user_list_1);
  new (&get_list(1)) list_type(user_list_2);
  /* be sure the two lists have the same size */
  if (get_list(0).size() != get_list(1).size()) throw invalid_list();
  /* reduce the lists to minimality */
  for (index_type index = 0; index < get_list_number(); ++index)
    get_list(index).make_minimal();
  /* check that the two lists have no overlap. Throw an exception for
     invalid list if a non-empty intersection is found. */
  if (get_list(0).is_overlapping(get_list(1))) throw invalid_list();
}

/* *********************************************************************
   | This method returns the highest formal address used in the cur-   |
   | rent slice + 1. It is sufficient to call the highest() method for |
   | each segment list in the slice. This number is the minimum size   |
   | of the register to be fed into this slice at execution time.      |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      24 Oct 2002         |
   ********************************************************************* */
Qop_slice::size_type Qop_slice::get_occupation(void) const {
  // reset the highest address sofar
  size_type highest_address = 0;
  // iterate through all lists in this slice
  for (index_type index = 0; index < get_list_number(); ++index) {
    // get the highest address
    size_type list_highest_address = get_list(index).highest();
    // update the overall value if necessary
    if (list_highest_address > highest_address)
      highest_address = list_highest_address;
  }
  // return the highest address found + 1 (since addresses start from 0)
  return (highest_address + 1);
}

/* *********************************************************************
   | This method implements a modification of the permutation list in  |
   | the quantum interface. It is used as a replacement for the execu- |
   | tion method by swap time slices. Actually, the only routine which |
   | is run here is the first index to address translation, then the   |
   | control is taken by the quantum interface.                        |
   | ----------------------------------------------------------------- |
   | If the list collection generated by translate() does not contain  |
   | exactly two lists, an exception is issued. This however should    |
   | not happen!                                                       |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 03 Dec 2001         |
   ********************************************************************* */
void Qop_slice::classical_permutation(const Qreg &a_register) const {
  /* run the index to address translation */
  const Qubit_list_collection swap_lists = translate(a_register);
  /* be sure that there are only two lists! */
  if (swap_lists.size() != 2) throw invalid_list();
  /* send the translated addresses */
  get_quantum_interface().submit_permutation(swap_lists[0], swap_lists[1]);
}

/* *********************************************************************
   | This method takes a quantum register (containing real addresses)  |
   | and runs the index to address translations with the internal      |
   | lists used as index lists. This means that for each list L in the |
   | time slice, a new list T is produced which contains some of the   |
   | addresses of the register list R: the i-th element of T is the    |
   | L_i-th element of R.                                              |
   | The translations are returned as a collection_type object.        |
   | (30 Nov 2001) S.Bettelli, moved mixing of lists to Qinterface.    |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 01 Aug 2001         |
   ********************************************************************* */
Qop_slice::collection_type Qop_slice::translate(const Qreg &a_register) const {
  /* This is the container to be returned at the end of the routine,
     carrying the translated lists. Put it here to facilitate the
     named return value optimisation. The size of the container is
     the number of index lists. */
  collection_type translated_lists(get_list_number());
  /* for each internal list, run the translation algorithm. In order to
     be sure that there are no unnecessary calls to copy ctors, swap
     the returned list into the corresponding slot in the container. */
  for (index_type index = 0; index < get_list_number(); ++index) {
    list_type translation_of_current_list =
      get_list(index).translate(a_register.get_qubit_list());
    translated_lists[index].swap(translation_of_current_list);
  }
  /* return the container */
  return translated_lists;
}

/* *********************************************************************
   | This method returns true if the passed slice and the current      |
   | slice are one the adjoint of the other. This requires that the    |
   | following conditions are met:                                     |
   |   1) the underlying gates are adjoint                             |
   |   2) the index lists are equivalent                               |
   | ----------------------------------------------------------------- |
   | Item 2) This test returns true if the corresponding lists of the  |
   |  two slices are independently equivalent, i.e. if for each pair   |
   |  of corresponding lists (l1,l2) the test l1.is_equivalent(l2) is  |
   |  true. This part of the routine should indeed be more elaborated. |
   |  Infact it can miss some "equivalent" sets of fixed arity opera-  |
   |  tion lists, since the order of the first list is meaningless.    |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 04 Jan 2002         |
   ********************************************************************* */
bool Qop_slice::is_adjoint_of(const Qop_slice &a_slice) const {
  /* test condition 1 and return immediately if it is not met. 
     If it is met, then it is guaranteed that there is the same
     number of internal index lists (two basegates are adjoint
     only if they have the same dimension). */
  if (! has_adjoint_basegate(a_slice)) return false;
  /* now loop over all internal index lists and test corresponding
     lists for equivalence. I know that this is not perfect, but it
     is currently enough. */
  for (index_type index = 0; index < get_list_number(); ++index)
    if (! get_list(index).is_equivalent(a_slice.get_list(index)))
      return false;
  /* if all tests were OK, return true. */
  return true;
}

/* *********************************************************************
   | This method returns "true" if the supplied address is contained   |
   | in any of the address lists of this slice, "false" otherwise.     |
   |                                                                   |
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      11 Nov 2002         |
   ********************************************************************* */
bool Qop_slice::is_referencing(address an_address) const {
  /* test the address against each list. Return with "true" as soon as
     the test is positive. Use is_overlapping() between a list and a
     segment (maybe this could be optimised even more?) */
  for (index_type index = 0; index < get_list_number(); ++index)
    if (get_list(index).is_overlapping(Qubit_segment(an_address,1)))
      return true;
  /* if you get here, all the tests failed. Return false. */
  return false;
}

/* *********************************************************************
   | This output function will display the "content" of a time slice.  |
   | First, it prints the pointer, the operation code, a human-readable|
   | description of the gate and the number of index lists. Additional |
   | slice-specific details can then be supplied through a virtual     |
   | method. Then all the index lists are shown.                       |
   | ----------------------------------------------------------------- |
   | Since the passed argument cannot be a real slice (slices can't be |
   | instantiated), it must be the base part of a derived class: this  |
   | class can provided more details through virtual get_details().    |
   |                                                                   |
   | Stefano Bettelli, INFN and Trento University, 24 May 2001         | 
   | Stefano Bettelli, IRSAMC, UPS, Toulouse,      11 Nov 2002         |
   ********************************************************************* */
std::ostream &operator<<(std::ostream &os, const Qop_slice &a_slice) {
  /* print some slice details which don't rely on virtual methods. */
  os << "Slice " << &a_slice << " opcode = "         // slice identifier
     << std::setw(4) << a_slice.get_opcode() << " [" // operation code
     << std::setw(9) << a_slice.get_name() << "]"    // operation name
     << " [N=" << a_slice.get_list_number() << "]";  // # of index lists
  /* print additional slice details using the virtual table. */
  a_slice.get_details(os);
  /* print out all the internal lists. */
  for (Qop_slice::index_type index = 0;
       index < a_slice.get_list_number(); ++index)
    os << "\n" << a_slice.get_list(index);
  /* return the stream */
  return os;
}

//;;; Local Variables: ***
//;;; mode:C++ ***
//;;; End: ***
